<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Pharmacy extends Admin_Controller
{

    public function __construct()
    {
        parent::__construct();
        $this->config->load("payroll");
        $this->load->library('Enc_lib');
        $this->load->library('mailsmsconf');
        $this->load->library('encoding_lib');
        $this->load->library('CSVReader');
        $this->load->model('pharmacy_model');
        $this->load->model('expensehead_model');
        $this->load->model('expense_model');
        $this->load->model('medicine_category_model');
        $this->marital_status       = $this->config->item('marital_status');
        $this->payment_mode         = $this->config->item('payment_mode');
        $this->search_type          = $this->config->item('search_type');
        $this->blood_group          = $this->config->item('bloodgroup');
        $this->charge_type          = $this->customlib->getChargeMaster();
        $data["charge_type"]        = $this->charge_type;
        $this->patient_login_prefix = "pat";
        $this->config->load("image_valid");
    }

    public function unauthorized()
    {
        $data = array();
        $this->load->view('layout/header', $data);
        $this->load->view('unauthorized', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules(
            'medicine_name',
            $this->lang->line('medicine') . " " . $this->lang->line('name'),
            array(
                'required',
                array('check_exists', array($this->medicine_category_model, 'valid_medicine_name')),
            )
        );
        $this->form_validation->set_rules('medicine_category_id', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'required');
        $this->form_validation->set_rules('medicine_company', $this->lang->line('medicine') . " " . $this->lang->line('company'), 'required');
        $this->form_validation->set_rules('medicine_composition', $this->lang->line('medicine') . " " . $this->lang->line('composition'), 'required');
        $this->form_validation->set_rules('medicine_group', $this->lang->line('medicine') . " " . $this->lang->line('group'), 'required');
        $this->form_validation->set_rules('unit', $this->lang->line('unit'), 'required');
        $this->form_validation->set_rules('unit_packing', $this->lang->line('unit') . "/" . $this->lang->line('packing'), 'required');
        $this->form_validation->set_rules('file', $this->lang->line('image'), 'callback_handle_upload', 'required');
        $this->form_validation->set_rules('barcode', 'Barcode', 'required');
        if ($this->form_validation->run() == false) {
            $msg = array(
                'medicine_name'        => form_error('medicine_name'),
                'medicine_category_id' => form_error('medicine_category_id'),
                'medicine_company'     => form_error('medicine_company'),
                'medicine_composition' => form_error('medicine_composition'),
                'medicine_group'       => form_error('medicine_group'),
                'unit'                 => form_error('unit'),
                'unit_packing'         => form_error('unit_packing'),
                'file'                 => form_error('file'),
                'barcode'              => form_error('barcode'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $pharmacy = array(
                'medicine_name' => $this->input->post('medicine_name'),
                'medicine_category_id'            => $this->input->post('medicine_category_id'),
                'medicine_company'                => $this->input->post('medicine_company'),
                'medicine_composition'            => $this->input->post('medicine_composition'),
                'medicine_group'                  => $this->input->post('medicine_group'),
                'unit'                            => $this->input->post('unit'),
                'min_level'                       => $this->input->post('min_level'),
                'reorder_level'                   => $this->input->post('reorder_level'),
                'vat'                             => $this->input->post('vat'),
                'barcode'                         => $this->input->post('barcode'),
                'unit_packing'                    => $this->input->post('unit_packing'),
                'supplier'                        => $this->input->post('supplier'),
                'note'                            => $this->input->post('note'),
                'vat_ac'                          => $this->input->post('vat_ac'),
            );
            $insert_id = $this->pharmacy_model->add($pharmacy);

            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $fileInfo = pathinfo($_FILES["file"]["name"]);
                $img_name = $insert_id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/medicine_images/" . $img_name);
                $data_img = array('id' => $insert_id, 'medicine_image' => 'uploads/medicine_images/' . $img_name);
                $this->pharmacy_model->update($data_img);
            }
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function search()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $medicineCategory         = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"] = $medicineCategory;
        $resultlist               = $this->pharmacy_model->searchFullText();
        $i                        = 0;
        foreach ($resultlist as $value) {
            $pharmacy_id                 = $value['id'];
            $available_qty               = $this->pharmacy_model->totalQuantity($pharmacy_id);
            $totalAvailableQty           = $available_qty['total_qty'];
            $resultlist[$i]["total_qty"] = $totalAvailableQty;
            $i++;
        }
        $result             = $this->pharmacy_model->getPharmacy();
        $data['resultlist'] = $resultlist;
        $data['result']     = $result;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/search', $data);
        $this->load->view('layout/footer');
    }

    public function dt_search()
    {

        $draw            = $_POST['draw'];
        $row             = $_POST['start'];
        $rowperpage      = $_POST['length']; // Rows display per page
        $columnIndex     = $_POST['order'][0]['column']; // Column index
        $columnName      = $_POST['columns'][$columnIndex]['data']; // Column name
        $columnSortOrder = $_POST['order'][0]['dir']; // asc or desc
        $where_condition = array();
        if (!empty($_POST['search']['value'])) {
            $where_condition = array('search' => $_POST['search']['value']);
        }
        $resultlist   = $this->pharmacy_model->search_datatable($where_condition);
        // echo $this->db->last_query();exit;
        $total_result = $this->pharmacy_model->search_datatable_count($where_condition);
        $data         = array();
        $total_qty = 0;
        $total_purchase = 0;
        $total_sale = 0;
        foreach ($resultlist as $result_key => $result_value) {
            $status = "";
            if ($result_value->total_qty <= 0) {

                $status = " <span class='text text-danger'> (" . $this->lang->line('out_of_stock') . ")</span>";
            } elseif ($result_value->total_qty <= $result_value->min_level) {

                $status = " <span class='text text-warning'> (" . $this->lang->line('low_stock') . ")</span>";
            } else if ($result_value->total_qty <= $result_value->reorder_level) {

                $status = "";
                $status = " <span class='text text-info'> (" . $this->lang->line('reorder') . ")</span>";
            }

            $nestedData = array();
            $action     = "<div class='rowoptionview'>";
            $action .= "<a href='#' onclick='viewDetail(" . $result_value->id . ")' class='btn btn-default btn-xs' data-toggle='tooltip' title='" . $this->lang->line('show') . "' ><i class='fa fa-reorder'></i></a>";
            if ($this->rbac->hasPrivilege('medicine_bad_stock', 'can_add')) {
                $action .= "<a href='#' class='btn btn-default btn-xs' onclick='addbadstock(" . $result_value->id . ")' data-toggle='tooltip' title='" . $this->lang->line('add') . ' ' . $this->lang->line('bad') . ' ' . $this->lang->line('stock') . "' > <i class='fas fa-minus-square'></i> </a>";
            }

            $total_qty += $result_value->total_qty;
            $total_purchase += $result_value->total_purchase;
            $total_sale += $result_value->total_sale;

            $action .= "<div'>";

            $nestedData[] = $result_value->medicine_name . $action;
            $nestedData[] = $result_value->medicine_company;
            $nestedData[] = $result_value->medicine_composition;
            $nestedData[] = $result_value->medicine_category;
            $nestedData[] = $result_value->medicine_group;
            $nestedData[] = $result_value->unit;
            $nestedData[] = $result_value->total_qty . $status;
            $nestedData[] = $result_value->total_purchase;
            $nestedData[] = $result_value->total_sale;

            $data[]       = $nestedData;
        }

        $data[] = array(
            // Add empty cells for the columns preceding the totals
            '',
            '',
            '',
            '',
            '',
            'Total',
            // Add the totals
            $total_qty,
            round($total_purchase, 2),
            round($total_sale, 2)
        );

        //Total Records Calculation
        $totalRecords   = $this->pharmacy_model->search_pharmacy($where_condition);
        $available_qty = 0;
        $available_purchase = 0;
        $available_sale = 0;
        foreach ($totalRecords as $key => $value) {
            $available_qty += $value->total_qty;
            $available_purchase += $value->total_purchase;
            $available_sale += $value->total_sale;
        }

        $json_data = array(
            "draw"            => intval($draw), // for every request/draw by clientside , they send a number as a parameter, when they recieve a response/data they first check the draw number, so we are sending same number in draw.
            "recordsTotal"    => intval($total_result), // total number of records
            "recordsFiltered" => intval($total_result), // total number of records after searching, if there is no searching then totalFiltered = totalData
            "total_qty"       => $available_qty,
            "total_purchase"  => round($available_purchase, 2),
            "total_sale"      => round($available_sale, 2),
            "data"            => $data, // total data array
        );

        echo json_encode($json_data); // send data as json format

    }

    public function bill_search()
    {

        $draw            = $_POST['draw'];
        $row             = $_POST['start'];
        $rowperpage      = $_POST['length']; // Rows display per page
        $columnIndex     = $_POST['order'][0]['column']; // Column index
        $columnName      = $_POST['columns'][$columnIndex]['data']; // Column name
        $columnSortOrder = $_POST['order'][0]['dir']; // asc or desc
        $where_condition = array();
        if (!empty($_POST['search']['value'])) {
            $where_condition = array('search' => $_POST['search']['value']);
        }
        $resultlist   = $this->pharmacy_model->searchbill_datatable($where_condition);
        $total_result = $this->pharmacy_model->searchbill_datatable_count($where_condition);
        $data         = array();
        foreach ($resultlist as $result_key => $result_value) {

            $nestedData = array();
            $action     = "<div class='rowoptionview'>";
            $action .= "<a href='#' onclick='viewDetail(" . $result_value->id . "," . $result_value->bill_no . "," . $result_value->patient_id . ")' class='btn btn-default btn-xs' data-toggle='tooltip' title='" . $this->lang->line('show') . "' ><i class='fa fa-reorder'></i></a>";
            $action .= "<a href='#' onclick='viewDetail(" . $result_value->id . "," . $result_value->bill_no . "," . $result_value->patient_id . ")' class='btn btn-default btn-xs' data-toggle='tooltip' title='" . $this->lang->line('print') . "' ><i class='fa fa-print'></i></a>";
            $action .= "<a href='#' onclick='refund_bill(" . $result_value->id . "," . $result_value->bill_no . "," . $result_value->patient_id . ")' class='btn btn-default btn-xs' data-toggle='tooltip' title='Refund'>
                <i class='fa fa-undo'></i>
            </a>";
            $action .= "<a href='" . site_url('admin/pharmacy/refundList/' . $result_value->id) . "' class='btn btn-default btn-xs' data-toggle='tooltip' title='" . $this->lang->line('refund') . ' ' . $this->lang->line('list') . "'>
                <i class='fa fa-list'></i>
            </a>";

            $action .= "<div'>";

            $nestedData[] = $result_value->bill_no;
            $nestedData[] = $result_value->date;
            $nestedData[] = $result_value->patient_name . $action;
            $nestedData[] = $result_value->doctor_name;
            $nestedData[] = $result_value->net_amount;
            $data[]       = $nestedData;
        }

        $json_data = array(
            "draw"            => intval($draw), // for every request/draw by clientside , they send a number as a parameter, when they recieve a response/data they first check the draw number, so we are sending same number in draw.
            "recordsTotal"    => intval($total_result), // total number of records
            "recordsFiltered" => intval($total_result), // total number of records after searching, if there is no searching then totalFiltered = totalData
            "data"            => $data, // total data array
        );

        echo json_encode($json_data); // send data as json format

    }
    //======================================================================================
    public function handle_upload()
    {
        $image_validate = $this->config->item('image_validate');
        if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {

            $file_type         = $_FILES["file"]['type'];
            $file_size         = $_FILES["file"]["size"];
            $file_name         = $_FILES["file"]["name"];
            $allowed_extension = $image_validate['allowed_extension'];
            $ext               = pathinfo($file_name, PATHINFO_EXTENSION);
            $allowed_mime_type = $image_validate['allowed_mime_type'];
            if ($files = @getimagesize($_FILES['file']['tmp_name'])) {

                if (!in_array($files['mime'], $allowed_mime_type)) {
                    $this->form_validation->set_message('handle_upload', $this->lang->line('file_type_not_allowed'));
                    return false;
                }
                if (!in_array(strtolower($ext), $allowed_extension) || !in_array($file_type, $allowed_mime_type)) {
                    $this->form_validation->set_message('handle_upload', $this->lang->line('file_type_not_allowed'));
                    return false;
                }
                if ($file_size > $image_validate['upload_size']) {
                    $this->form_validation->set_message('handle_upload', $this->lang->line('file_size_shoud_be_less_than') . number_format($image_validate['upload_size'] / 1048576, 2) . " MB");
                    return false;
                }
            } else {
                $this->form_validation->set_message('handle_upload', $this->lang->line('file_type_not_allowed'));
                return false;
            }

            return true;
        }
        return true;
    }

    public function check_upload()
    {
        $image_validate = $this->config->item('image_validate');
        if (isset($_FILES["medicine_image"]) && !empty($_FILES['medicine_image']['name'])) {

            $file_type         = $_FILES["medicine_image"]['type'];
            $file_size         = $_FILES["medicine_image"]["size"];
            $file_name         = $_FILES["medicine_image"]["name"];
            $allowed_extension = $image_validate['allowed_extension'];
            $ext               = pathinfo($file_name, PATHINFO_EXTENSION);
            $allowed_mime_type = $image_validate['allowed_mime_type'];
            if ($files = @getimagesize($_FILES['medicine_image']['tmp_name'])) {

                if (!in_array($files['mime'], $allowed_mime_type)) {
                    $this->form_validation->set_message('check_upload', $this->lang->line('file_type_not_allowed'));
                    return false;
                }
                if (!in_array($ext, $allowed_extension) || !in_array($file_type, $allowed_mime_type)) {
                    $this->form_validation->set_message('check_upload', $this->lang->line('file_type_not_allowed'));
                    return false;
                }
                if ($file_size > $image_validate['upload_size']) {
                    $this->form_validation->set_message('check_upload', $this->lang->line('file_size_shoud_be_less_than') . number_format($image_validate['upload_size'] / 1048576, 2) . " MB");
                    return false;
                }
            } else {
                $this->form_validation->set_message('check_upload', $this->lang->line('file_type_not_allowed'));
                return false;
            }

            return true;
        }
        return true;
    }

    public function getindate()
    {
        $purchase_id = $this->input->post("purchase_id");
        $result      = $this->pharmacy_model->getindate($purchase_id);
        echo json_encode($result);
    }

    public function getdate()
    {
        $id             = $this->input->post("id");
        $result         = $this->pharmacy_model->getdate($id);
        $result["date"] = date($this->customlib->getSchoolDateFormat(true, true), strtotime($result['date']));

        echo json_encode($result);
    }
    public function purchase()
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $medicineCategory         = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"] = $medicineCategory;
        $supplierCategory         = $this->medicine_category_model->getSupplierCategory();
        $data["supplierCategory"] = $supplierCategory;

        $resultlist = $this->pharmacy_model->getSupplier();
        $i          = 0;
        foreach ($resultlist as $value) {
            $pharmacy_id                 = $value['id'];
            $available_qty               = $this->pharmacy_model->totalQuantity($pharmacy_id);
            $totalAvailableQty           = $available_qty['total_qty'];
            $resultlist[$i]["total_qty"] = $totalAvailableQty;
            $i++;
        }
        $result             = $this->pharmacy_model->getPharmacy();
        $data['resultlist'] = $resultlist;
        $data['result']     = $result;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/purchase.php', $data);
        $this->load->view('layout/footer');
    }

    public function returnStock()
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $medicineCategory         = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"] = $medicineCategory;
        $supplierCategory         = $this->medicine_category_model->getSupplierCategory();
        $data["supplierCategory"] = $supplierCategory;

        // Get all supplier purchases for dropdown
        $purchases = $this->pharmacy_model->getSupplier();
        $data['purchases'] = $purchases;

        // Get return list
        $resultlist = $this->pharmacy_model->getSupplierReturn();
        $data['resultlist'] = $resultlist;

        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/returnStock.php', $data);
        $this->load->view('layout/footer');
    }

    public function getPurchasesBySupplier()
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $supplier_id = $this->input->post("supplier_id");
        $this->db->select('supplier_bill_basic.id, supplier_bill_basic.purchase_no, supplier_bill_basic.invoice_no, supplier_bill_basic.date, supplier_bill_basic.net_amount');
        $this->db->where('supplier_bill_basic.supplier_id', $supplier_id);
        $this->db->order_by('supplier_bill_basic.id', 'DESC');
        $query = $this->db->get('supplier_bill_basic');
        $result = $query->result_array();
        echo json_encode($result);
    }

    public function exportformat()
    {
        $this->load->helper('download');
        $filepath = "./backend/import/import_medicine_sample_file.csv";
        $data     = file_get_contents($filepath);
        $name     = 'import_medicine_sample_file.csv';

        force_download($name, $data);
    }

    public function import()
    {
        if (!$this->rbac->hasPrivilege('import_medicine', 'can_view')) {
            access_denied();
        }
        $this->form_validation->set_rules('medicine_category_id', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'required|trim|xss_clean');

        $this->form_validation->set_rules('file', $this->lang->line('file'), 'callback_handle_csv_upload');
        $medicineCategory         = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"] = $medicineCategory;
        $fields                   = array('medicine_name', 'medicine_company', 'medicine_composition', 'medicine_group', 'unit', 'min_level', 'reorder_level', 'vat', 'unit_packing', 'note', 'barcode');
        $data["fields"]           = $fields;

        if ($this->form_validation->run() == false) {
            $msg = array(
                'medicine_category_id' => form_error('medicine_category_id'),
                'file'                 => form_error('file'),
            );

            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
            $this->load->view('layout/header');
            $this->load->view('admin/pharmacy/import', $data);
            $this->load->view('layout/footer');
        } else {
            $medicine_category_id = $this->input->post('medicine_category_id');
            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $ext = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);

                if ($ext == 'csv') {
                    $file = $_FILES['file']['tmp_name'];

                    $result = $this->csvreader->parse_file($file);
                    if (!empty($result)) {
                        $count = 0;
                        for ($i = 1; $i <= count($result); $i++) {

                            $medicine_data[$i] = array();
                            $n                 = 0;
                            foreach ($result[$i] as $key => $value) {

                                $medicine_data[$i][$fields[$n]]            = $this->encoding_lib->toUTF8($result[$i][$key]);
                                $medicine_data[$i]['is_active']            = 'yes';
                                $medicine_data[$i]['medicine_category_id'] = $medicine_category_id;

                                $n++;
                            }

                            $medicine_name = $medicine_data[$i]["medicine_name"];
                            if (!empty($medicine_name)) {
                                if ($this->pharmacy_model->check_medicine_exists($medicine_name, $medicine_category_id)) {
                                    $this->session->set_flashdata('msg', '<div class="alert alert-danger text-center">' . $this->lang->line('record_already_exists') . '</div>');

                                    $insert_id = "";
                                } else {
                                    $insert_id = $this->pharmacy_model->addImport($medicine_data[$i]);
                                }
                            }

                            if (!empty($insert_id)) {
                                $data['csvData'] = $result;
                                $this->session->set_flashdata('msg', '<div class="alert alert-success text-center">' . $this->lang->line('students_imported_successfully') . '</div>');
                                $count++;
                                $this->session->set_flashdata('msg', '<div class="alert alert-success text-center">Total ' . count($result) . " records found in CSV file. Total " . $count . ' records imported successfully.</div>');
                            } else {

                                $this->session->set_flashdata('msg', '<div class="alert alert-danger text-center">' . $this->lang->line('record_already_exists') . '</div>');
                            }
                        }
                    }
                }
                redirect('admin/pharmacy/import');
            }
        }
    }

    public function importBulk()
    {
        if (!$this->rbac->hasPrivilege('import_medicine', 'can_view')) {
            access_denied();
        }

        $this->form_validation->set_rules('file', $this->lang->line('file'), 'callback_handle_csv_upload');
        $medicineCategory         = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"] = $medicineCategory;

        if ($this->form_validation->run() == false) {
            $msg = array(
                'file' => form_error('file'),
            );

            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
            $this->load->view('layout/header');
            $this->load->view('admin/pharmacy/import_bulk', $data);
            $this->load->view('layout/footer');
        } else {
            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $ext = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);

                if ($ext == 'csv') {
                    $file = $_FILES['file']['tmp_name'];

                    // Custom CSV parsing to handle metadata rows and detect delimiter
                    $result = $this->parseBulkImportCSV($file);

                    if (!empty($result)) {
                        $errors = array();
                        $success_count = 0;
                        $purchase_bills = array(); // Group by supplier
                        $category_cache = array(); // Cache category IDs
                        $supplier_cache = array(); // Cache supplier IDs

                        // Get first available category as default (if category is missing in CSV)
                        $default_category_id = null;
                        if (!empty($medicineCategory) && isset($medicineCategory[0])) {
                            $default_category_id = $medicineCategory[0]['id'];
                        }

                        // Process each row (parseBulkImportCSV already handles header detection)
                        for ($i = 1; $i <= count($result); $i++) {
                            if (!isset($result[$i])) {
                                continue;
                            }

                            $row = $result[$i];

                            // Get column values (case-insensitive, handle various formats)
                            $item_name = $this->getColumnValue($row, array('Item Name', 'item name', 'item_name', 'ITEM NAME', 'ItemName', 'itemName'));
                            $generic_name = $this->getColumnValue($row, array('Genaric Name', 'generic name', 'generic_name', 'Generic Name', 'GENERIC NAME', 'GenericName'));
                            $category_name = $this->getColumnValue($row, array('Category', 'category', 'CATEGORY', 'Category'));
                            $manufacturer = $this->getColumnValue($row, array('Manufacturer', 'manufacturer', 'MANUFACTURER', 'Manufacturer'));
                            $supplier_str = $this->getColumnValue($row, array('Supplier(s)', 'Supplier', 'supplier(s)', 'supplier', 'SUPPLIER', 'SUPPLIER(S)', 'Suppliers'));
                            $unit_cost = $this->getColumnValue($row, array('Unit Cost', 'unit cost', 'unit_cost', 'Unit Cost', 'UNIT COST', 'UnitCost'));
                            $retail_value = $this->getColumnValue($row, array('Retail Value', 'retail value', 'retail_value', 'Retail Value', 'RETAIL VALUE', 'RetailValue'));
                            $ordering_level = $this->getColumnValue($row, array('Ordering level', 'ordering level', 'ordering_level', 'Ordering level', 'ORDERING LEVEL', 'OrderingLevel', 'Ordering Level'));
                            $available_qty = $this->getColumnValue($row, array('Available Qty.', 'Available Qty', 'available qty', 'available_qty', 'Available Qty.', 'AVAILABLE QTY', 'AvailableQty', 'Available Qty'));
                            $rack_no = $this->getColumnValue($row, array('Rack#', 'rack#', 'rack', 'Rack#', 'RACK#', 'Rack', 'Rack No', 'rack no'));

                            // Skip empty rows
                            if (empty($item_name)) {
                                continue;
                            }

                            // Validate required fields
                            if (empty($item_name)) {
                                $errors[] = "Row " . ($i + 1) . ": Item Name is required";
                                continue;
                            }

                            // Get or create category (use default if empty)
                            $category_id = null;
                            if (!empty($category_name)) {
                                $category_id = $this->getOrCreateCategory($category_name, $category_cache);
                            }

                            // If category not found or empty, use default category
                            if (!$category_id && $default_category_id) {
                                $category_id = $default_category_id;
                            }

                            // If still no category, skip this row
                            if (!$category_id) {
                                $errors[] = "Row " . ($i + 1) . ": No category available. Please create at least one category first.";
                                continue;
                            }

                            // Process supplier (treat comma-separated values as one supplier name)
                            $primary_supplier_id = null;
                            if (!empty($supplier_str)) {
                                // Don't split by comma - treat the whole string as one supplier name
                                $supplier_name = trim($supplier_str);
                                $primary_supplier_id = $this->getOrCreateSupplier($supplier_name, $supplier_cache);
                            }

                            // Create or get medicine
                            $medicine_data = array(
                                'medicine_name' => $this->encoding_lib->toUTF8($item_name),
                                'medicine_company' => $this->encoding_lib->toUTF8($manufacturer),
                                'medicine_composition' => $this->encoding_lib->toUTF8($generic_name),
                                'medicine_group' => '',
                                'unit' => '',
                                'min_level' => !empty($ordering_level) ? $ordering_level : 0,
                                'reorder_level' => !empty($ordering_level) ? $ordering_level : 0,
                                'vat' => 0,
                                'unit_packing' => 1,
                                'note' => !empty($rack_no) ? 'Rack: ' . $rack_no : '',
                                'is_active' => 'yes',
                                'medicine_category_id' => $category_id,
                            );

                            // Check if medicine exists
                            $medicine_id = null;
                            if ($this->pharmacy_model->check_medicine_exists($medicine_data['medicine_name'], $category_id)) {
                                // Get existing medicine ID
                                $existing = $this->pharmacy_model->getMedicineByNameAndCategory($medicine_data['medicine_name'], $category_id);
                                if ($existing) {
                                    $medicine_id = $existing['id'];
                                }
                            } else {
                                $medicine_id = $this->pharmacy_model->addImport($medicine_data);
                            }

                            if (!$medicine_id) {
                                $errors[] = "Row " . ($i + 1) . ": Failed to create medicine: " . $item_name;
                                continue;
                            }

                            // If we have supplier and quantities, create purchase bill entry
                            if ($primary_supplier_id && !empty($available_qty) && $available_qty > 0) {
                                $purchase_price = !empty($unit_cost) ? floatval($unit_cost) : 0;
                                $sale_price = !empty($retail_value) ? floatval($retail_value) : 0;
                                $quantity = !empty($available_qty) ? intval($available_qty) : 0;

                                if (!isset($purchase_bills[$primary_supplier_id])) {
                                    $purchase_bills[$primary_supplier_id] = array(
                                        'supplier_id' => $primary_supplier_id,
                                        'items' => array()
                                    );
                                }

                                $purchase_bills[$primary_supplier_id]['items'][] = array(
                                    'pharmacy_id' => $medicine_id,
                                    'medicine_category_id' => $category_id,
                                    'purchase_price' => $purchase_price,
                                    'sale_rate' => $sale_price,
                                    'quantity' => $quantity,
                                    'available_quantity' => $quantity,
                                    'amount' => $purchase_price * $quantity,
                                );
                            }

                            $success_count++;
                        }

                        // Create purchase bills
                        $purchase_count = 0;
                        $bill_date = date('Y-m-d H:i:s');

                        foreach ($purchase_bills as $supplier_id => $bill_data) {
                            $purchase_no = $this->pharmacy_model->getMaxId();
                            if (empty($purchase_no)) {
                                $purchase_no = 0;
                            }
                            $purchase_no = $purchase_no + 1;

                            // Calculate totals
                            $total = 0;
                            foreach ($bill_data['items'] as $item) {
                                $total += $item['amount'];
                            }

                            // Create purchase bill
                            $bill_basic = array(
                                'date' => $bill_date,
                                'supplier_id' => $supplier_id,
                                'supplier_name' => '',
                                'invoice_no' => 'BULK-IMPORT-' . date('YmdHis') . '-' . $purchase_no,
                                'purchase_no' => $purchase_no,
                                'total' => $total,
                                'discount' => 0,
                                'tax' => 0,
                                'net_amount' => $total,
                                'note' => 'Bulk Import - ' . date('Y-m-d H:i:s'),
                            );

                            $bill_id = $this->pharmacy_model->addBillSupplier($bill_basic);

                            if ($bill_id) {
                                // Add bill details
                                $bill_details = array();
                                foreach ($bill_data['items'] as $item) {
                                    $bill_details[] = array(
                                        'inward_date' => $bill_date,
                                        'pharmacy_id' => $item['pharmacy_id'],
                                        'supplier_bill_basic_id' => $bill_id,
                                        'medicine_category_id' => $item['medicine_category_id'],
                                        'expiry_date' => '',
                                        'expiry_date_format' => null,
                                        'batch_no' => '',
                                        'batch_amount' => 0,
                                        'mrp' => $item['sale_rate'],
                                        'sale_rate' => $item['sale_rate'],
                                        'packing_qty' => 1,
                                        'quantity' => $item['quantity'],
                                        'purchase_price' => $item['purchase_price'],
                                        'available_quantity' => $item['available_quantity'],
                                        'amount' => $item['amount'],
                                    );
                                }

                                $this->pharmacy_model->addBillMedicineBatchSupplier($bill_details);
                                $purchase_count++;
                            }
                        }

                        // Set success message
                        $message = "Total " . $success_count . " medicines imported successfully. ";
                        if ($purchase_count > 0) {
                            $message .= $purchase_count . " purchase bill(s) created.";
                        }
                        if (!empty($errors)) {
                            $message .= " " . count($errors) . " error(s) occurred.";
                        }

                        $this->session->set_flashdata('msg', '<div class="alert alert-success text-center">' . $message . '</div>');

                        if (!empty($errors)) {
                            $error_msg = '<div class="alert alert-warning text-center"><strong>Errors:</strong><ul>';
                            foreach (array_slice($errors, 0, 20) as $error) {
                                $error_msg .= '<li>' . $error . '</li>';
                            }
                            if (count($errors) > 20) {
                                $error_msg .= '<li>... and ' . (count($errors) - 20) . ' more errors</li>';
                            }
                            $error_msg .= '</ul></div>';
                            $this->session->set_flashdata('error_details', $error_msg);
                        }
                    } else {
                        $this->session->set_flashdata('msg', '<div class="alert alert-danger text-center">No data found in CSV file.</div>');
                    }
                } else {
                    $this->session->set_flashdata('msg', '<div class="alert alert-danger text-center">Invalid file format. Please upload a CSV file.</div>');
                }
            }

            // Check if AJAX request
            if ($this->input->is_ajax_request()) {
                if (isset($message)) {
                    $array = array('status' => 'success', 'error' => '', 'message' => $message);
                } else {
                    $array = array('status' => 'success', 'error' => '', 'message' => 'Import completed.');
                }
                echo json_encode($array);
            } else {
                redirect('admin/pharmacy/importBulk');
            }
        }
    }

    private function parseBulkImportCSV($file_path)
    {
        $result = array();
        $handle = fopen($file_path, 'r');

        if ($handle === false) {
            return $result;
        }

        // Detect delimiter by reading first few lines
        $first_lines = '';
        $line_count = 0;
        $temp_handle = fopen($file_path, 'r');
        while ($line_count < 10 && ($line = fgets($temp_handle)) !== false) {
            $first_lines .= $line;
            $line_count++;
        }
        fclose($temp_handle);

        $comma_count = substr_count($first_lines, ',');
        $semicolon_count = substr_count($first_lines, ';');
        $delimiter = ($comma_count > $semicolon_count) ? ',' : ';';

        // Find header row (look for "Item Name" and "Category" columns)
        $header_row_index = -1;
        $header_keys = array();
        $row_index = 0;

        // Reset file pointer
        rewind($handle);

        while (($row = fgetcsv($handle, 4096, $delimiter, '"')) !== false) {
            if (empty($row) || (count($row) == 1 && trim($row[0]) == '')) {
                continue;
            }

            // Check if this row contains header columns
            $row_lower = array_map('strtolower', array_map('trim', $row));
            $has_item_name = in_array('item name', $row_lower) || in_array('itemname', $row_lower);
            $has_category = in_array('category', $row_lower);

            if ($has_item_name && $has_category) {
                $header_row_index = $row_index;
                // Store header keys with original case
                foreach ($row as $key) {
                    $header_keys[] = trim($key);
                }
                break;
            }
            $row_index++;
        }

        // If header not found, use first row as header
        if ($header_row_index == -1) {
            rewind($handle);
            $first_row = fgetcsv($handle, 4096, $delimiter, '"');
            if ($first_row) {
                $header_keys = array_map('trim', $first_row);
                $header_row_index = 0;
            }
        }

        // Read data rows (skip header row)
        $data_index = 1;
        rewind($handle);
        $current_row = 0;

        while (($row = fgetcsv($handle, 4096, $delimiter, '"')) !== false) {
            if ($current_row <= $header_row_index) {
                $current_row++;
                continue;
            }

            if (empty($row) || (count($row) == 1 && trim($row[0]) == '')) {
                $current_row++;
                continue;
            }

            // Map row data to header keys
            $row_data = array();
            foreach ($header_keys as $index => $key) {
                if (!empty($key)) {
                    $value = isset($row[$index]) ? trim($row[$index]) : '';
                    $row_data[$key] = $this->encoding_lib->toUTF8($value);
                }
            }

            // Only add if row has some data
            if (!empty($row_data) && !empty(array_filter($row_data))) {
                $result[$data_index] = $row_data;
                $data_index++;
            }

            $current_row++;
        }

        fclose($handle);
        return $result;
    }

    private function getColumnValue($row, $possible_keys)
    {
        foreach ($possible_keys as $key) {
            if (isset($row[$key]) && !empty($row[$key])) {
                return trim($row[$key]);
            }
        }
        return '';
    }

    private function getOrCreateCategory($category_name, &$cache)
    {
        $category_name = trim($category_name);
        if (empty($category_name)) {
            return null;
        }

        // Check cache first
        if (isset($cache[$category_name])) {
            return $cache[$category_name];
        }

        // Check if exists in database
        $this->db->where('medicine_category', $category_name);
        $query = $this->db->get('medicine_category');
        if ($query->num_rows() > 0) {
            $category = $query->row_array();
            $cache[$category_name] = $category['id'];
            return $category['id'];
        }

        // Create new category
        $data = array('medicine_category' => $category_name);
        $category_id = $this->medicine_category_model->addMedicineCategory($data);
        if ($category_id) {
            $cache[$category_name] = $category_id;
            return $category_id;
        }

        return null;
    }

    private function getOrCreateSupplier($supplier_name, &$cache)
    {
        $supplier_name = trim($supplier_name);
        if (empty($supplier_name)) {
            return null;
        }

        // Check cache first
        if (isset($cache[$supplier_name])) {
            return $cache[$supplier_name];
        }

        // Check if exists in database
        $this->db->where('supplier_category', $supplier_name);
        $query = $this->db->get('supplier_category');
        if ($query->num_rows() > 0) {
            $supplier = $query->row_array();
            $cache[$supplier_name] = $supplier['id'];
            return $supplier['id'];
        }

        // Create new supplier
        $data = array(
            'supplier_category' => $supplier_name,
            'contact' => '',
            'supplier_person' => '',
            'supplier_person_contact' => '',
            'address' => '',
        );
        $supplier_id = $this->medicine_category_model->addSupplierCategory($data);
        if ($supplier_id) {
            $cache[$supplier_name] = $supplier_id;
            return $supplier_id;
        }

        return null;
    }

    public function handle_csv_upload()
    {

        $image_validate = $this->config->item('filecsv_validate');
        if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {

            $file_type         = $_FILES["file"]['type'];
            $file_size         = $_FILES["file"]["size"];
            $file_name         = $_FILES["file"]["name"];
            $allowed_extension = $image_validate['allowed_extension'];
            $ext               = pathinfo($file_name, PATHINFO_EXTENSION);
            $allowed_mime_type = $image_validate['allowed_mime_type'];
            if ($files = filesize($_FILES['file']['tmp_name'])) {

                if (!in_array($file_type, $allowed_mime_type)) {
                    $this->form_validation->set_message('handle_csv_upload', 'File Type Not Allowed');
                    return false;
                }

                if (!in_array($ext, $allowed_extension) || !in_array($file_type, $allowed_mime_type)) {
                    $this->form_validation->set_message('handle_csv_upload', 'Extension Not Allowed');
                    return false;
                }
                if ($file_size > $image_validate['upload_size']) {
                    $this->form_validation->set_message('handle_csv_upload', $this->lang->line('file_size_shoud_be_less_than') . number_format($image_validate['upload_size'] / 1048576, 2) . " MB");
                    return false;
                }
            } else {
                $this->form_validation->set_message('handle_csv_upload', "File Type / Extension Not Allowed");
                return false;
            }

            return true;
        } else {
            $this->form_validation->set_message('handle_csv_upload', "File field is required");
            return false;
        }
        return true;
    }

    public function getDetails()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $id     = $this->input->post("pharmacy_id");
        $result = $this->pharmacy_model->getDetails($id);
        echo json_encode($result);
    }

    public function update()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_edit')) {
            access_denied();
        }
        $this->form_validation->set_rules('medicine_name', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_category_id', $this->lang->line('medicine') . " " . $this->lang->line('category') . " " . $this->lang->line('id'), 'required');
        $this->form_validation->set_rules('medicine_company', $this->lang->line('medicine') . " " . $this->lang->line('company'), 'required');
        $this->form_validation->set_rules('medicine_composition', $this->lang->line('medicine') . " " . $this->lang->line('composition'), 'required');
        $this->form_validation->set_rules('medicine_group', $this->lang->line('medicine') . " " . $this->lang->line('group'), 'required');
        $this->form_validation->set_rules('unit', $this->lang->line('unit'), 'required');
        $this->form_validation->set_rules('unit_packing', $this->lang->line('unit') . "/" . $this->lang->line('packing'), 'required');
        $this->form_validation->set_rules('medicine_image', $this->lang->line('image'), 'callback_check_upload');
        $this->form_validation->set_rules('barcode', 'Barcode', 'required');
        if ($this->form_validation->run() == false) {
            $msg = array(
                'medicine_name'        => form_error('medicine_name'),
                'medicine_category_id' => form_error('medicine_category_id'),
                'medicine_company'     => form_error('medicine_company'),
                'medicine_composition' => form_error('medicine_composition'),
                'medicine_group'       => form_error('medicine_group'),
                'unit'                 => form_error('unit'),
                'unit_packing'         => form_error('unit_packing'),
                'medicine_image'       => form_error('medicine_image'),
                'barcode'              => form_error('barcode'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $id       = $this->input->post('id');
            $pharmacy = array(
                'id'                   => $id,
                'medicine_name'        => $this->input->post('medicine_name'),
                'medicine_category_id' => $this->input->post('medicine_category_id'),
                'medicine_company'     => $this->input->post('medicine_company'),
                'medicine_composition' => $this->input->post('medicine_composition'),
                'medicine_group'       => $this->input->post('medicine_group'),
                'unit'                 => $this->input->post('unit'),
                'min_level'            => $this->input->post('min_level'),
                'reorder_level'        => $this->input->post('reorder_level'),
                'vat'                  => $this->input->post('vat'),
                'unit_packing'         => $this->input->post('unit_packing'),
                'note'                 => $this->input->post('edit_note'),
                'vat_ac'               => $this->input->post('vat_ac'),
                'barcode'              => $this->input->post('barcode'),
            );
            $this->pharmacy_model->update($pharmacy);
            if (isset($_FILES["medicine_image"]) && !empty($_FILES['medicine_image']['name'])) {
                $fileInfo = pathinfo($_FILES["medicine_image"]["name"]);
                $img_name = $id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["medicine_image"]["tmp_name"], "./uploads/medicine_images/" . $img_name);
                $data_img = array('id' => $id, 'medicine_image' => 'uploads/medicine_images/' . $img_name);
                $this->pharmacy_model->update($data_img);
            }
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function delete($id)
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_delete')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->pharmacy_model->delete($id);
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }

    public function getPharmacy()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $id     = $this->input->post('pharmacy_id');
        $result = $this->pharmacy_model->getPharmacy($id);
        echo json_encode($result);
    }

    public function convertMonthToNumber($monthName)
    {
        return date('m', strtotime($monthName));
    }

    public function medicineBatch()
    {
        if (!$this->rbac->hasPrivilege('medicine batch details', 'can_edit')) {
            access_denied();
        }
        $this->form_validation->set_rules('pharmacy_id', $this->lang->line('pharmacy') . " " . $this->lang->line('id'), 'required');
        $this->form_validation->set_rules('expiry_date', $this->lang->line('expiry') . " " . $this->lang->line('date'), 'required');
        $this->form_validation->set_rules('batch_no', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');
        $this->form_validation->set_rules('packing_qty', $this->lang->line('packing') . " " . $this->lang->line('qty'), 'required|numeric');
        $this->form_validation->set_rules('quantity', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('mrp', $this->lang->line('mrp'), 'required|numeric');
        $this->form_validation->set_rules('sale_rate', $this->lang->line('sale') . " " . $this->lang->line('rate'), 'required|numeric');

        if ($this->form_validation->run() == false) {
            $msg = array(
                'pharmacy_id'        => form_error('pharmacy_id'),
                'expiry_date'        => form_error('expiry_date'),
                'expiry_date_format' => form_error('expiry_date_format'),
                'batch_no'           => form_error('batch_no'),
                'packing_qty'        => form_error('packing_qty'),
                'quantity'           => form_error('quantity'),
                'mrp'                => form_error('mrp'),
                'sale_rate'          => form_error('sale_rate'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $id          = $this->input->post('pharmacy_id');
            $inward_date = $this->input->post('inward_date');
            $expdate     = $this->input->post('expiry_date');

            $explore = explode("/", $expdate);

            $monthary = $explore[0];
            $yearary  = $explore[1];
            $month    = $monthary;

            $month_number = $this->convertMonthToNumber($month);
            $insert_date  = $yearary . "-" . $month_number . "-01";

            $medicine_batch = array(
                'pharmacy_id'           => $id,
                'expiry_date'           => $this->input->post('expiry_date'),
                'expiry_date_format'    => $insert_date,
                'inward_date'           => date('Y-m-d', $this->customlib->datetostrtotime($inward_date)),
                'batch_no'              => $this->input->post('batch_no'),
                'packing_qty'           => $this->input->post('packing_qty'),
                'purchase_rate_packing' => $this->input->post('purchase_rate_packing'),
                'quantity'              => $this->input->post('quantity'),
                'mrp'                   => $this->input->post('mrp'),
                'sale_rate'             => $this->input->post('sale_rate'),
                'amount'                => $this->input->post('amount'),
                'available_quantity'    => $this->input->post('quantity'),
            );
            $this->pharmacy_model->medicineDetail($medicine_batch);
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function getMedicineBatch()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $id                     = $this->input->post("pharmacy_id");
        $result                 = $this->pharmacy_model->getMedicineBatch($id);
        $data["result"]         = $result;
        $badstockresult         = $this->pharmacy_model->getMedicineBadStock($id);
        $data["badstockresult"] = $badstockresult;

        $this->load->view('admin/pharmacy/medicineDetail', $data);
    }

    public function addpatient()
    {
        if (!$this->rbac->hasPrivilege('patient', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('name', $this->lang->line('name'), 'trim|required|xss_clean');

        if ($this->form_validation->run() == false) {
            $msg = array(
                'name' => form_error('name'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $check_patient_id = $this->patient_model->getMaxId();

            if (empty($check_patient_id)) {
                $check_patient_id = 1000;
            }

            $patient_id = $check_patient_id + 1;

            $patient_data = array(
                'patient_name'      => $this->input->post('name'),
                'mobileno'          => $this->input->post('contact'),
                'marital_status'    => $this->input->post('marital_status'),
                'email'             => $this->input->post('email'),
                'gender'            => $this->input->post('gender'),
                'guardian_name'     => $this->input->post('guardian_name'),
                'blood_group'       => $this->input->post('blood_group'),
                'address'           => $this->input->post('address'),
                'known_allergies'   => $this->input->post('known_allergies'),
                'patient_unique_id' => $patient_id,
                'note'              => $this->input->post('note'),
                'age'               => $this->input->post('age'),
                'month'             => $this->input->post('month'),
                'is_active'         => 'yes',
            );
            $insert_id = $this->patient_model->add_patient($patient_data);

            $user_password      = $this->role->get_random_password($chars_min = 6, $chars_max = 6, $use_upper_case = false, $include_numbers = true, $include_special_chars = false);
            $data_patient_login = array(
                'username' => $this->patient_login_prefix . $insert_id,
                'password' => $user_password,
                'user_id'  => $insert_id,
                'role'     => 'patient',
            );
            $this->user_model->add($data_patient_login);
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $fileInfo = pathinfo($_FILES["file"]["name"]);
                $img_name = $insert_id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/patient_images/" . $img_name);
                $data_img = array('id' => $insert_id, 'image' => 'uploads/patient_images/' . $img_name);
                $this->patient_model->add($data_img);
            }
        }
        echo json_encode($array);
    }

    public function patientDetails()
    {

        if (!$this->rbac->hasPrivilege('patient', 'can_view')) {
            access_denied();
        }
        $id   = $this->input->post("id");
        $data = $this->patient_model->patientDetails($id);
        echo json_encode($data);
    }

    public function supplierDetails()
    {

        if (!$this->rbac->hasPrivilege('medicine_supplier', 'can_view')) {
            access_denied();
        }
        $id   = $this->input->post("id");
        $data = $this->patient_model->supplierDetails($id);
        echo json_encode($data);
    }

    public function bill()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'pharmacy');
        $doctors                  = $this->staff_model->getStaffbyrole(3);
        $data["doctors"]          = $doctors;
        $data['medicineCategory'] = $this->medicine_category_model->getMedicineCategory();
        $data['medicineName']     = $this->pharmacy_model->getMedicineName();
        // $patients                 = $this->patient_model->getPatientListall();
        // $data["patients"]         = $patients;
        $data["marital_status"]   = $this->marital_status;
        $data["bloodgroup"]       = $this->blood_group;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/pharmacyBill.php', $data);
        $this->load->view('layout/footer');
    }

    public function getAllPatients()
    {
        $search = $this->input->get('search');
        $page = $this->input->get('page') ?: 1;
        $perPage = 10;
        $offset = ($page - 1) * $perPage;


        $patients = $this->patient_model->search_patients($search, $perPage, $offset);
        $total = $this->patient_model->count_patients($search);

        $results = [];
        foreach ($patients as $dvalue) {

            $check = isset($dvalue->patient_cnic) && !empty($dvalue->patient_cnic) ? " (" . $dvalue->patient_cnic . ")" : '';
            $text = $dvalue->patient_name . '(' . $dvalue->patient_unique_id . ') (' . $dvalue->mobileno . ') ' . $check;

            $results[] = ['id' => $dvalue->id, 'text' => $text];
        }

        echo json_encode([
            'results' => $results,
            'pagination' => ['more' => ($offset + $perPage) < $total]
        ]);
    }

    public function get_medicine_name()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $medicine_category_id = $this->input->post("medicine_category_id");
        $data                 = $this->pharmacy_model->get_medicine_name($medicine_category_id);
        echo json_encode($data);
    }

    public function get_medicine_dosage()
    {

        $medicine_category_id = $this->input->post("medicine_category_id");
        $data                 = $this->pharmacy_model->get_medicine_dosage($medicine_category_id);
        echo json_encode($data);
    }
    public function get_medicine_instruction()
    {
        //echo "am here";exit;
        $medicine_category_id = $this->input->post("medicine_category_id");
        $data                 = $this->medicine_dosage_model->getMedicineInstruction($id = null, $medicine_category_id);
        echo json_encode($data);
    }

    public function get_supplier_name()
    {
        if (!$this->rbac->hasPrivilege('supplier_category', 'can_view')) {
            access_denied();
        }
        $supplier_category_id = $this->input->post("supplier_category_id");
        $data                 = $this->pharmacy_model->get_supplier_name($supplier_category_id);
        echo json_encode($data);
    }

    public function addBill()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_id[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        // $this->form_validation->set_rules('batch_no[]', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');

        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('sale_price[]', $this->lang->line('sale_price'), 'required|numeric');
        $this->form_validation->set_rules('amount[]', $this->lang->line('amount'), 'required|numeric');
        $this->form_validation->set_rules('total', $this->lang->line('total'), 'required|numeric');
        $this->form_validation->set_rules('patient_id', $this->lang->line('patient'), 'trim|required|xss_clean');
        if ($this->form_validation->run() == false) {
            $msg = array(
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_id'        => form_error('medicine_id[]'),
                // 'batch_no'             => form_error('batch_no[]'),
                'quantity'             => form_error('quantity[]'),
                'sale_price'           => form_error('sale_price[]'),
                'amount'               => form_error('amount[]'),
                'total'                => form_error('total'),
                'patient_id'           => form_error('patient_id'),
            );

            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $patient_id = $this->input->post('patient_id');
            $bill_date = $this->input->post("date");
            $bill_no   = $this->input->post('bill_no');

            $data = array(
                'bill_no'       => $bill_no,
                'date'          => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                'patient_id'    => $patient_id,
                'customer_name' => $this->input->post('customer_name'),
                'doctor_name'   => $this->input->post('doctor_name'),
                'total'         => $this->input->post('total'),
                'discount'      => $this->input->post('discount'),
                'tax'           => $this->input->post('tax'),
                'net_amount'    => $this->input->post('net_amount'),
                'note'          => $this->input->post('note'),
                'generated_by'  => $this->session->userdata('hospitaladmin')['id'],
                'created_at'    => date('Y-m-d H:i:s', strtotime($bill_date)),
            );

            $bill_id = $this->pharmacy_model->addBill($data);

            if ($bill_id) {
                $pharmacy_bill_basic_id    = $bill_id;
                $medicine_category_id      = $this->input->post('medicine_category_id');
                $medicine_name             = $this->input->post('medicine_id');
                $expire_date               = $this->input->post('expire_date');
                $batch_no                  = $this->input->post('batch_no');
                $quantity                  = $this->input->post('quantity');
                $total_quantity            = $this->input->post('available_quantity');
                $medicine_batch_details_id = $this->input->post('id');
                $sale_price                = $this->input->post('sale_price');
                $amount                    = $this->input->post('amount');
                $discount_single           = $this->input->post('discount_single');
                $data                      = array();
                $i                         = 0;
                foreach ($medicine_category_id as $key => $value) {
                    $detail = array(
                        'pharmacy_bill_basic_id' => $bill_id,
                        'medicine_category_id'   => $medicine_category_id[$i],
                        'medicine_name'          => $medicine_name[$i],
                        'expire_date'            => $expire_date[$i],
                        'batch_no'               => isset($batch_no[$i]) && !empty($batch_no[$i]) ? $batch_no[$i] : '',
                        'quantity'               => $quantity[$i],
                        'sale_price'             => $sale_price[$i],
                        'amount'                 => $amount[$i],
                        'discount_single'        => $discount_single[$i],
                    );
                    $available_quantity[$i] = $total_quantity[$i] - $quantity[$i];
                    $update_quantity        = array(
                        'id'                 => $medicine_batch_details_id[$i],
                        'available_quantity' => $available_quantity[$i],
                    );
                    $data[] = $detail;
                    $this->pharmacy_model->availableQty($update_quantity);
                    $i++;
                }

                $this->pharmacy_model->addBillBatch($data);
            } else {
            }
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'), 'insert_id' => $bill_id);
        }
        echo json_encode($array);
    }

    public function getBillDetails($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $print_details         = $this->printing_model->get('', 'pharmacy');
        $data["print_details"] = $print_details;
        $data['id']            = $id;
        if (isset($_POST['print'])) {
            $data["print"] = 'yes';
        } else {
            $data["print"] = 'no';
        }

        $result         = $this->pharmacy_model->getBillDetails($id);
        $data['result'] = $result;
        $detail         = $this->pharmacy_model->getAllBillDetails($id);
        $data['detail'] = $detail;
        // echo "<pre>"; print_r($data);exit;
        $this->load->view('admin/pharmacy/printBill', $data);
    }

    public function getSupplierDetails($id)
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $data['id'] = $id;
        if (isset($_POST['print'])) {
            $data["print"] = 'yes';
        } else {
            $data["print"] = 'no';
        }

        $result         = $this->pharmacy_model->getSupplierDetails($id);
        $data['result'] = $result;
        $detail         = $this->pharmacy_model->getAllSupplierDetails($id);
        $data['detail'] = $detail;
        $this->load->view('admin/pharmacy/printPurchase', $data);
    }

    public function download($file)
    {

        $this->load->helper('download');

        $filepath = "./uploads/medicine_images/" . $this->uri->segment(6);
        $data     = file_get_contents($filepath);
        $name     = $this->uri->segment(6);
        force_download($name, $data);
    }

    public function getQuantity()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $batch_no = $this->input->get('batch_no');
        $med_id   = $this->input->get('med_id');
        $data     = $this->pharmacy_model->getQuantity($batch_no, $med_id);
        echo json_encode($data);
    }

    public function getQuantityedit()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        $batch_no = $this->input->get('batch_no');
        $data = $this->pharmacy_model->getQuantityedit($batch_no);
        echo json_encode($data);
    }

    public function billReport()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'Reports');
        $this->session->set_userdata('sub_menu', 'admin/pharmacy/billreport');
        $select = 'pharmacy_bill_basic.*,patients.patient_name';
        $join   = array(
            'LEFT JOIN patients ON pharmacy_bill_basic.patient_id = patients.id',
        );

        $table_name  = "pharmacy_bill_basic";
        $search_type = $this->input->post("search_type");
        if (isset($search_type)) {
            $search_type = $this->input->post("search_type");
        } else {
            $search_type = "this_month";
        }
        if (empty($search_type)) {

            $search_type = "";
            $resultlist  = $this->report_model->getReport($select, $join, $table_name, $where = array());
        } else {

            $search_table  = "pharmacy_bill_basic";
            $search_column = "date";
            $resultlist    = $this->report_model->searchReport($select, $join, $table_name, $search_type, $search_table, $search_column, $where = array());
        }
        $data["searchlist"]  = $this->search_type;
        $data["search_type"] = $search_type;
        $data["resultlist"]  = $resultlist;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/billReport.php', $data);
        $this->load->view('layout/footer');
    }

    public function editPharmacyBill($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $medicineCategory             = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"]     = $medicineCategory;
        $medicine_category_id         = $this->input->post("medicine_category_id");
        $data['medicine_category_id'] = $this->pharmacy_model->get_medicine_name($medicine_category_id);
        $data['medicine_category_id'] = $medicine_category_id;
        $patients                     = $this->patient_model->getPatientListall();
        $data["patients"]             = $patients;
        $doctors                      = $this->staff_model->getStaffbyrole(3);
        $data["doctors"]              = $doctors;
        $result                       = $this->pharmacy_model->getBillDetails($id);
        $data['result']               = $result;
        $detail                       = $this->pharmacy_model->getAllBillDetails($id);
        $data['detail']               = $detail;
        $this->load->view("admin/pharmacy/editPharmacyBill", $data);
    }
    public function refundPharmacyBill($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $medicineCategory             = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"]     = $medicineCategory;
        $medicine_category_id         = $this->input->post("medicine_category_id");
        $data['medicine_category_id'] = $this->pharmacy_model->get_medicine_name($medicine_category_id);
        $data['medicine_category_id'] = $medicine_category_id;
        $patients                     = $this->patient_model->getPatientListall();
        $data["patients"]             = $patients;
        $doctors                      = $this->staff_model->getStaffbyrole(3);
        $data["doctors"]              = $doctors;
        $result                       = $this->pharmacy_model->getBillDetails($id);
        $data['result']               = $result;
        $detail                       = $this->pharmacy_model->getAllBillDetails($id);
        $data['detail']               = $detail;
        $this->load->view("admin/pharmacy/refundPharmacyBill", $data);
    }

    public function editPharmaBill($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }

        $id = $this->input->post("id");
        $patients         = $this->patient_model->getPatientListall();
        $data["patients"] = $patients;
        $result         = $this->pharmacy_model->getBillDetails($id);
        $data['result'] = $result;
        echo json_encode($result);
    }

    public function editSupplierBill($id)
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $medicineCategory             = $this->medicine_category_model->getMedicineCategory();
        $data["medicineCategory"]     = $medicineCategory;
        $medicine_category_id         = $this->input->post("medicine_category_id");
        $data['medicine_category_id'] = $this->pharmacy_model->get_medicine_name($medicine_category_id);
        $data['medicine_category_id'] = $medicine_category_id;
        $supplierCategory             = $this->medicine_category_model->getSupplierCategory();
        $data["supplierCategory"]     = $supplierCategory;
        $supplier_category_id         = $this->input->post("supplier_category_id");
        $data['supplier_category_id'] = $this->pharmacy_model->get_supplier_name($supplier_category_id);
        $data['supplier_category_id'] = $supplier_category_id;

        $result         = $this->pharmacy_model->getSupplierDetails($id);
        $data['result'] = $result;
        $detail         = $this->pharmacy_model->getAllSupplierDetails($id);
        $data['detail'] = $detail;
        $this->load->view("admin/pharmacy/editSupplierBill", $data);
    }

    public function updateBill()
    {

        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_edit')) {
            access_denied();
        }

        $this->form_validation->set_rules('bill_no', $this->lang->line('bill') . " " . $this->lang->line('no'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('patient_id', $this->lang->line('patient'), 'trim|required');
        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_name[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('expire_date[]', $this->lang->line('expire') . " " . $this->lang->line('date'), 'required');
        $this->form_validation->set_rules('batch_no[]', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');
        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('sale_price[]', $this->lang->line('sale_price'), 'required|numeric');
        $this->form_validation->set_rules('amount[]', $this->lang->line('amount'), 'required|numeric');
        $this->form_validation->set_rules('total', $this->lang->line('total'), 'required|numeric');
        if ($this->form_validation->run() == false) {

            $msg = array(
                'bill_no'              => form_error('bill_no'),
                'date'                 => form_error('date'),
                'customer_name'        => form_error('customer_name'),
                'patient_id'           => form_error('patient_id'),
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_name'        => form_error('medicine_name[]'),
                'expire_date'          => form_error('expire_date[]'),
                'batch_no'             => form_error('batch_no[]'),
                'quantity'             => form_error('quantity[]'),
                'sale_price'           => form_error('sale_price[]'),
                'total'                => form_error('total'),
                'amount'               => form_error('amount[]'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {

            $id               = $this->input->post('bill_basic_id');
            $bill_id          = $this->input->post("bill_detail_id[]");
            $previous_bill_id = $this->input->post("previous_bill_id[]");
            $bill_date        = $this->input->post("date");
            $data_array       = array();
            $delete_arr       = array();
            foreach ($previous_bill_id as $pkey => $pvalue) {
                if (in_array($pvalue, $bill_id)) {
                } else {
                    $delete_arr[] = array('id' => $pvalue);
                }
            }

            $data = array(
                'id'            => $id,
                'bill_no'       => $this->input->post('bill_no'),
                'patient_id'    => $this->input->post('patient_id'),
                'date'          => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                'customer_name' => $this->input->post('customer_name'),
                'customer_type' => $this->input->post('customer_type'),
                'doctor_name'   => $this->input->post('doctor_name'),
                'opd_ipd_no'    => $this->input->post('opd_ipd_no'),
                'total'         => $this->input->post('total'),
                'discount'      => $this->input->post('discount'),
                'note'          => $this->input->post('note'),
                'tax'           => $this->input->post('tax'),
                'net_amount'    => $this->input->post('net_amount'),
            );

            $this->pharmacy_model->addBill($data);

            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $fileInfo = pathinfo($_FILES["file"]["name"]);
                $img_name = $id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/pres_images/" . $img_name);
                $data_img = array('id' => $id, 'file' => 'uploads/pres_images/' . $img_name);
                $this->pharmacy_model->addBill($data_img);
            }

            if (!empty($id)) {
                $pharmacy_bill_basic_id = $id;
                $bill_detail_id         = $this->input->post('bill_detail_id');
                $medicine_batch_id      = $this->input->post('medicine_batch_id');
                $medicine_category_id   = $this->input->post('medicine_category_id');
                $medicine_name          = $this->input->post('medicine_name');
                $expiry_date            = $this->input->post('expire_date');
                $batch_no               = $this->input->post('batch_no');
                $quantity               = $this->input->post('quantity');
                $total_quantity         = $this->input->post('available_quantity');
                $amount                 = $this->input->post('amount');
                $discount_single             = $this->input->post('discount_single');
                $sale_price             = $this->input->post('sale_price');
                $data                   = array();
                $i                      = 0;
                foreach ($medicine_category_id as $key => $value) {
                    if ($bill_id[$i] == 0) {
                        $add_data = array(
                            'pharmacy_bill_basic_id' => $id,
                            'medicine_category_id'   => $medicine_category_id[$i],
                            'medicine_name'          => $medicine_name[$i],
                            'expire_date'            => $expiry_date[$i],
                            'batch_no'               => $batch_no[$i],
                            'quantity'               => $quantity[$i],
                            'sale_price'             => $sale_price[$i],
                            'amount'                 => $amount[$i],
                            'discount_single'        => $discount_single[$i],
                        );
                        $data_array[]           = $add_data;
                        $available_quantity[$i] = $total_quantity[$i] - $quantity[$i];
                        $add_quantity           = array(
                            'id'                 => $medicine_batch_id[$i],
                            'available_quantity' => $available_quantity[$i],
                        );
                        $this->pharmacy_model->availableQty($add_quantity);
                    } else {
                        $detail = array(
                            'id'                     => $bill_detail_id[$i],
                            'pharmacy_bill_basic_id' => $id,
                            'medicine_category_id'   => $medicine_category_id[$i],
                            'medicine_name'          => $medicine_name[$i],
                            'expire_date'            => $expiry_date[$i],
                            'batch_no'               => $batch_no[$i],
                            'quantity'               => $quantity[$i],
                            'sale_price'             => $sale_price[$i],
                            'discount_single'        => $discount_single[$i],
                            'amount'                 => $amount[$i],
                        );
                        $this->pharmacy_model->updateBillDetail($detail);
                        $available_quantity[$i] = $total_quantity[$i] - $quantity[$i];
                        $update_quantity        = array(
                            'id'                 => $medicine_batch_id[$i],
                            'available_quantity' => $available_quantity[$i],
                        );
                        $this->pharmacy_model->availableQty($update_quantity);
                    }
                    $i++;
                }
            } else {
            }
            if (!empty($data_array)) {
                $this->pharmacy_model->addBillBatch($data_array);
            }
            if (!empty($delete_arr)) {
                $this->pharmacy_model->delete_bill_detail($delete_arr);
            }
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function addBillSupplier()
    {

        if (!$this->rbac->hasPrivilege('medicine_supplier', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('supplier_id', $this->lang->line('supplier'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_name[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('expiry_date[]', $this->lang->line('expiry') . " " . $this->lang->line('date'), 'required');
        // $this->form_validation->set_rules('batch_no[]', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');
        $this->form_validation->set_rules('mrp[]', $this->lang->line('mrp'), 'required');
        $this->form_validation->set_rules('sale_rate[]', $this->lang->line('sale_price'), 'required');
        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('purchase_price[]', $this->lang->line('purchase') . " " . $this->lang->line('price'), 'required|numeric');
        $this->form_validation->set_rules('amount[]', $this->lang->line('amount'), 'required|numeric');
        $this->form_validation->set_rules('total', $this->lang->line('total'), 'required|numeric');
        if ($this->form_validation->run() == false) {
            $msg = array(
                'date'                 => form_error('date'),
                'supplier_id'          => form_error('supplier_id'),
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_name'        => form_error('medicine_name[]'),
                // 'batch_no'             => form_error('batch_no[]'),
                'mrp'                  => form_error('mrp[]'),
                'sale_rate'            => form_error('sale_rate[]'),
                'expiry_date'          => form_error('expiry_date[]'),
                'quantity'             => form_error('quantity[]'),
                'purchase_price'       => form_error('purchase_price[]'),
                'total'                => form_error('total'),
                'amount'               => form_error('amount[]'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $supplier_id   = $this->input->post('supplier_id');
            $supplier_name = $this->input->post('supplier_name');
            $bill_date     = $this->input->post("date");
            $purchase_no   = $this->pharmacy_model->getMaxId();
            if (empty($purchase_no)) {
                $purchase_no = 0;
            }
            $purchase = $purchase_no + 1;

            $data = array(
                'date'          => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                'supplier_id'   => $supplier_id,
                'supplier_name' => $supplier_name,
                'invoice_no'    => $this->input->post('invoiceno'),
                'purchase_no'   => $purchase,
                'total'         => $this->input->post('total'),
                'discount'      => $this->input->post('discount'),
                'tax'           => $this->input->post('tax'),
                'net_amount'    => $this->input->post('net_amount'),
                'note'          => $this->input->post('note'),
            );

            $insert_id = $this->pharmacy_model->addBillSupplier($data);

            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $fileInfo = pathinfo($_FILES["file"]["name"]);
                $img_name = $insert_id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/medicine_images/" . $img_name);
                $data_img = array('id' => $insert_id, 'file' => 'uploads/medicine_images/' . $img_name);
                $this->pharmacy_model->addBillSupplier($data_img);
            }

            if ($insert_id) {
                $medicine_category_id = $this->input->post('medicine_category_id');
                $medicine_name        = $this->input->post('medicine_name');
                $expiry_date          = $this->input->post('expiry_date');
                $batch_no             = $this->input->post('batch_no');
                $batch_amount         = $this->input->post('batch_amount');
                $mrp                  = $this->input->post('mrp');
                $sale_rate            = $this->input->post('sale_rate');
                $packing_qty          = $this->input->post('packing_qty');
                $quantity             = $this->input->post('quantity');
                $purchase_price       = $this->input->post('purchase_price');
                $amount               = $this->input->post('amount');

                $data1 = array();
                $j     = 0;
                foreach ($medicine_name as $key => $mvalue) {

                    $expdate = $expiry_date[$j];

                    $explore = explode("/", $expdate);

                    $monthary = $explore[0];
                    $yearary  = $explore[1];
                    $month    = $monthary;

                    $month_number = $this->convertMonthToNumber($month);
                    $insert_date  = $yearary . "-" . $month_number . "-01";

                    $details = array(
                        'inward_date'            => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                        'pharmacy_id'            => $medicine_name[$j],
                        'supplier_bill_basic_id' => $insert_id,
                        'medicine_category_id'   => $medicine_category_id[$j],
                        'expiry_date'            => $expiry_date[$j],
                        'expiry_date_format'     => $insert_date,
                        'batch_no'               => isset($batch_no[$j]) && !empty($batch_no[$j]) ? $batch_no[$j] : '',
                        'batch_amount'           => $batch_amount[$j],
                        'mrp'                    => $mrp[$j],
                        'sale_rate'              => $sale_rate[$j],
                        'packing_qty'            => $packing_qty[$j],
                        'quantity'               => $quantity[$j],
                        'purchase_price'         => $purchase_price[$j],
                        'available_quantity'     => $quantity[$j],
                        'amount'                 => $amount[$j],
                    );
                    $data1[] = $details;

                    $j++;
                }

                $this->pharmacy_model->addBillMedicineBatchSupplier($data1);
            } else {
            }
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'), 'insert_id' => $insert_id);
        }
        echo json_encode($array);
    }

    public function updateSupplierBill()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_edit')) {
            access_denied();
        }

        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required|xss_clean');

        $this->form_validation->set_rules('supplier_id', $this->lang->line('supplier'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_name[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('expiry_date[]', $this->lang->line('expiry') . " " . $this->lang->line('date'), 'required');
        // $this->form_validation->set_rules('batch_no[]', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');
        $this->form_validation->set_rules('mrp[]', $this->lang->line('mrp'), 'required');
        $this->form_validation->set_rules('sale_rate[]', $this->lang->line('sale_price'), 'required');
        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('purchase_price[]', $this->lang->line('purchase_price'), 'required|numeric');
        $this->form_validation->set_rules('total', $this->lang->line('total'), 'required|numeric');
        if ($this->form_validation->run() == false) {
            $msg = array(
                'date'                 => form_error('date'),
                'supplier_id'          => form_error('supplier_id'),
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_name'        => form_error('medicine_name[]'),
                'expiry_date'          => form_error('expiry_date[]'),
                // 'batch_no'             => form_error('batch_no[]'),
                'mrp'                  => form_error('mrp[]'),
                'sale_rate'            => form_error('sale_rate[]'),
                'quantity'             => form_error('quantity[]'),
                'purchase_price'       => form_error('purchase_price[]'),
                'total'                => form_error('total'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $id               = $this->input->post('bill_basic_id');
            $bill_id          = $this->input->post("bill_detail_id[]");
            $previous_bill_id = $this->input->post("previous_bill_id[]");
            $supplier_id      = $this->input->post('supplier_id');
            $purchase_no      = $this->input->post('purchase_no');
            $data_array       = array();
            $delete_arr       = array();

            $bill_date = $this->input->post("date");
            $data      = array(
                'id'          => $id,
                'date'        => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                'supplier_id' => $supplier_id,
                'purchase_no' => $purchase_no,
                'invoice_no'  => $this->input->post('invoice_no'),
                'total'       => $this->input->post('total'),
                'discount'    => $this->input->post('discount'),
                'tax'         => $this->input->post('tax'),
                'note'        => $this->input->post('note'),
                'net_amount'  => $this->input->post('net_amount'),
            );

            if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                $fileInfo = pathinfo($_FILES["file"]["name"]);
                $img_name = $id . '.' . $fileInfo['extension'];
                move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/medicine_images/" . $img_name);
                $data_img = array('id' => $id, 'file' => 'uploads/medicine_images/' . $img_name);
                $this->pharmacy_model->addBillSupplier($data_img);
            }
            $this->pharmacy_model->addBillSupplier($data);

            if (!empty($id)) {

                $bill_detail_id       = $this->input->post('bill_detail_id');
                $medicine_batch_id    = $this->input->post('medicine_batch_id');
                $medicine_category_id = $this->input->post('medicine_category_id');
                $medicine_name        = $this->input->post('medicine_name');
                $expiry_date          = $this->input->post('expiry_date');
                $batch_no             = $this->input->post('batch_no');
                $batch_amount         = $this->input->post('batch_amount');
                $mrp                  = $this->input->post('mrp');
                $sale_rate            = $this->input->post('sale_rate');
                $packing_qty          = $this->input->post('packing_qty');
                $quantity             = $this->input->post('quantity');
                $total_quantity       = $this->input->post('available_quantity');
                $amount               = $this->input->post('amount');
                $purchase_price       = $this->input->post('purchase_price');
                $data_array1          = array();
                $bill_date1           = $this->input->post("date");
                $j                    = 0;
                foreach ($medicine_category_id as $key => $value) {

                    if ($bill_id[$j] == 0) {
                        $add_data = array(
                            'supplier_bill_basic_id' => $id,
                            'medicine_category_id'   => $medicine_category_id[$j],
                            'pharmacy_id'            => $medicine_name[$j],
                            'inward_date'            => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date1)),
                            'expiry_date'            => $expiry_date[$j],
                            'batch_no'               => isset($batch_no[$j]) && !empty($batch_no[$j]) ? $batch_no[$j] : '',
                            'batch_amount'           => $batch_amount[$j],
                            'mrp'                    => $mrp[$j],
                            'sale_rate'              => $sale_rate[$j],
                            'packing_qty'            => $packing_qty[$j],
                            'quantity'               => $quantity[$j],
                            'available_quantity'     => $quantity[$j],
                            'purchase_price'         => $purchase_price[$j],
                            'amount'                 => $amount[$j],
                        );
                        $data_array[] = $add_data;
                    } else {

                        $detail = array(
                            'id'                     => $bill_detail_id[$j],
                            'supplier_bill_basic_id' => $id,
                            'medicine_category_id'   => $medicine_category_id[$j],
                            'pharmacy_id'            => $medicine_name[$j],
                            'inward_date'            => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date1)),
                            'expiry_date'            => $expiry_date[$j],
                            'batch_no'               => $batch_no[$j],
                            'batch_amount'           => $batch_amount[$j],
                            'mrp'                    => $mrp[$j],
                            'sale_rate'              => $sale_rate[$j],
                            'packing_qty'            => $packing_qty[$j],
                            'quantity'               => $quantity[$j],
                            'available_quantity'     => $quantity[$j],
                            'purchase_price'         => $purchase_price[$j],
                            'amount'                 => $amount[$j],
                        );

                        $this->pharmacy_model->updateMedicineBatchDetail($detail);
                    }

                    $j++;
                }
            } else {
            }
            if (!empty($data_array)) {
                $this->pharmacy_model->addBillMedicineBatchSupplier($data_array);
            }

            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function deletePharmacyBill($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_delete')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->pharmacy_model->deletePharmacyBill($id);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Record Deleted Successfully.');
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }

    public function deleteSupplierBill($id)
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_delete')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->pharmacy_model->deleteSupplierBill($id);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Record Deleted Successfully.');
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }

    public function delete_medicine_batch($id)
    {
        if (!$this->rbac->hasPrivilege('medicine batch details', 'can_view')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->pharmacy_model->delete_medicine_batch($id);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Record Deleted Successfully.');
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }

    public function getBillNo()
    {
        $result = $this->pharmacy_model->getBillNo();

        $id = $result["id"];
        if (!empty($result["id"])) {
            $bill_no = $id + 1;
        } else {
            $bill_no = 1;
        }
        echo json_encode($bill_no);
    }

    public function getExpiryDate()
    {
        $batch_no = $this->input->get_post('batch_no');
        $med_id   = $this->input->get_post('med_id');
        $result   = $this->pharmacy_model->getExpiryDate($batch_no, $med_id);
        echo json_encode($result);
    }

    public function getExpireDate()
    {
        $batch_no = $this->input->get_post('batch_no');
        $result = $this->pharmacy_model->getExpireDate($batch_no);
        echo json_encode($result);
    }

    public function getBatchNoList()
    {
        $medicine = $this->input->get_post('medicine');
        $result   = $this->pharmacy_model->getBatchNoList($medicine);
        echo json_encode($result);
    }

    public function addBadStock()
    {
        if (!$this->rbac->hasPrivilege('medicine_bad_stock', 'can_view')) {
            access_denied();
        }
        $this->form_validation->set_rules('pharmacy_id', $this->lang->line('pharmacy') . " " . $this->lang->line('id'), 'required');
        $this->form_validation->set_rules('expiry_date', $this->lang->line('expire') . " " . $this->lang->line('date'), 'required');
        $this->form_validation->set_rules('batch_no', $this->lang->line('batch') . " " . $this->lang->line('no'), 'required');
        $this->form_validation->set_rules('packing_qty', $this->lang->line('qty'), 'required|numeric');
        $this->form_validation->set_rules('inward_date', $this->lang->line('out') . " " . $this->lang->line('ward') . " " . $this->lang->line('date'), 'required');

        if ($this->form_validation->run() == false) {
            $msg = array(
                'pharmacy_id' => form_error('pharmacy_id'),
                'expiry_date' => form_error('expiry_date'),
                'batch_no'    => form_error('batch_no'),
                'packing_qty' => form_error('packing_qty'),
                'inward_date' => form_error('inward_date'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $id          = $this->input->post('pharmacy_id');
            $inward_date = $this->input->post('inward_date');

            $medicine_batch = array(
                'pharmacy_id'  => $id,
                'expiry_date'  => $this->input->post('expiry_date'),
                'outward_date' => date('Y-m-d', $this->customlib->datetostrtotime($inward_date)),
                'batch_no'     => $this->input->post('batch_no'),
                'quantity'     => $this->input->post('packing_qty'),
                'note'         => $this->input->post('note'),
            );

            $batch_qty   = $this->input->post('available_quantity');
            $packing_qty = $this->input->post('packing_qty');

            if (!empty($batch_qty)) {
                $available_quantity = $batch_qty - $packing_qty;
            } else {
                $available_quantity = 0;
            }

            $update_data = array('id' => $this->input->post('medicine_batch_id'), 'available_quantity' => $available_quantity);

            $this->pharmacy_model->addBadStock($medicine_batch);
            $this->pharmacy_model->updateMedicineBatch($update_data);
            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }
        echo json_encode($array);
    }

    public function deleteBadStock($id)
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->pharmacy_model->deleteBadStock($id);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Record Deleted Successfully.');
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }

    public function profitloss()
    {

        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'Reports');
        $this->session->set_userdata('sub_menu', 'admin/pharmacy/profitloss');
        $select = 'pharmacy_bill_detail.*,medicine_batch_details.purchase_price,pharmacy_bill_basic.net_amount,pharmacy_bill_basic.bill_no,pharmacy_bill_basic.date,pharmacy.medicine_name as medicine';
        $join   = array(
            'INNER JOIN medicine_batch_details ON  pharmacy_bill_detail.batch_no = medicine_batch_details.batch_no
             INNER JOIN  pharmacy ON pharmacy.id = pharmacy_bill_detail.medicine_name
             INNER JOIN  pharmacy_bill_basic ON pharmacy_bill_detail.pharmacy_bill_basic_id = pharmacy_bill_basic.id',
        );

        $table_name  = "pharmacy_bill_detail";
        $search_type = $this->input->post("search_type");
        if (isset($search_type)) {
            $search_type = $this->input->post("search_type");
        } else {
            $search_type = "this_month";
        }
        if (empty($search_type)) {

            $search_type = "";
            $resultlist  = $this->report_model->getReportMedicine($select, $join, $table_name, $where = array());
        } else {
            $search_table  = "pharmacy_bill_basic";
            $search_column = "date";
            $resultlist    = $this->report_model->searchReportMedicine($select, $join, $table_name, $search_type, $search_table, $search_column, $where = array());
        }
        $data["searchlist"]  = $this->search_type;
        $data["search_type"] = $search_type;
        $data["resultlist"]  = $resultlist;
        // echo "<pre>";print_r($resultlist);exit;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/profitlost.php', $data);
        $this->load->view('layout/footer');
    }

    public function refundBill()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_edit')) {
            access_denied();
        }

        // Validation rules for refund form
        $this->form_validation->set_rules('bill_no', $this->lang->line('bill') . " " . $this->lang->line('no'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('patient_id', $this->lang->line('patient'), 'trim|required');
        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_name[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('amount[]', $this->lang->line('amount'), 'required|numeric');

        if ($this->form_validation->run() == false) {
            $msg = array(
                'bill_no'              => form_error('bill_no'),
                'date'                 => form_error('date'),
                'patient_id'           => form_error('patient_id'),
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_name'        => form_error('medicine_name[]'),
                'quantity'             => form_error('quantity[]'),
                'amount'               => form_error('amount[]'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $bill_basic_id = $this->input->post('bill_basic_id');
            $bill_date = $this->input->post('date');

            // Create refund record
            $refund_data = array(
                'refund_no'                => $this->input->post('bill_no'),
                'patient_id'               => $this->input->post('patient_id'),
                'pharmacy_bill_basic_id'   => $bill_basic_id,
                'refund_date'              => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                'refund_total'             => $this->input->post('net_amount'),
                'refund_note'              => $this->input->post('note')
            );

            $refund_id = $this->pharmacy_model->addRefundBill($refund_data);

            // Process refund items
            if (!empty($bill_basic_id)) {
                $bill_detail_id    = $this->input->post('bill_detail_id');
                $medicine_batch_id = $this->input->post('medicine_batch_id');
                $medicine_category_id = $this->input->post('medicine_category_id');
                $medicine_name     = $this->input->post('medicine_name');
                $quantity          = $this->input->post('quantity');
                $available_qty     = $this->input->post('available_quantity');
                $amount            = $this->input->post('amount');
                $sale_price        = $this->input->post('sale_price');
                $discount_single   = $this->input->post('discount_single');

                $refund_details = array();
                $total_refund_amount = 0;
                $total_refund_discount = 0;
                $original_total = (float)$this->input->post('total');
                $original_tax_percent = 0;
                $original_discount_percent = 0;

                for ($i = 0; $i < count($medicine_category_id); $i++) {
                    // Refunded amounts for this item
                    $refunded_qty = (float)$quantity[$i];
                    $refunded_amount = (float)$amount[$i];

                    // Add to refund detail table
                    $refund_details[] = array(
                        'pharmacy_bill_refund_id' => $refund_id,
                        'medicine_category_id'    => $medicine_category_id[$i],
                        'medicine_name'           => $medicine_name[$i],
                        'refunded_qty'            => $refunded_qty,
                        'sale_price'              => $sale_price[$i],
                        'refund_amount'           => $refunded_amount,
                        'pharmacy_bill_detail_id' => $bill_detail_id[$i],
                    );

                    // Update medicine batch available quantity (add back to stock)
                    $new_available_qty = (float)$available_qty[$i] + $refunded_qty;
                    $this->pharmacy_model->availableQty(array(
                        'id'                 => $medicine_batch_id[$i],
                        'available_quantity' => $new_available_qty,
                    ));

                    // Load existing bill detail to compute remaining values
                    $orig_detail = $this->db->get_where('pharmacy_bill_detail', array('id' => $bill_detail_id[$i]))->row_array();
                    $orig_qty = isset($orig_detail['quantity']) ? (float)$orig_detail['quantity'] : 0;
                    $orig_amount = isset($orig_detail['amount']) ? (float)$orig_detail['amount'] : ($orig_qty * (float)$sale_price[$i]);

                    // Refunded qty comes from the form
                    // Remaining quantity = original quantity - refunded quantity
                    $remaining_qty = max(0, $orig_qty - $refunded_qty);
                    // Recalculate remaining amount based on sale price
                    $remaining_amount = round($remaining_qty * (float)$sale_price[$i], 2);

                    $bill_detail_update = array(
                        'id'           => $bill_detail_id[$i],
                        'quantity'     => $remaining_qty,  // remaining qty after refund
                        'amount'       => $remaining_amount,  // remaining amount after refund
                        'refunded_qty' => $refunded_qty  // track refunded quantity (model will increment)
                    );
                    $this->pharmacy_model->updateBillDetail($bill_detail_update);

                    // Track totals for bill adjustment
                    $total_refund_amount += $refunded_amount;
                    $total_refund_discount += isset($discount_single[$i]) ? (float)$discount_single[$i] : 0;
                }

                // Batch insert refund details
                if (!empty($refund_details)) {
                    $this->pharmacy_model->addRefundDetailBill($refund_details);
                }

                // Calculate new bill totals based on refunded amounts
                // Get current bill details to calculate percentages
                $current_bill = $this->pharmacy_model->getBillDetails($bill_basic_id);
                $current_total = (float)$current_bill['total'];
                $current_discount = (float)$current_bill['discount'];
                $current_tax = (float)$current_bill['tax'];
                $current_net = (float)$current_bill['net_amount'];

                // Calculate percentage ratios if original total exists
                if ($current_total > 0) {
                    $discount_ratio = $current_discount / $current_total;
                    $tax_ratio = $current_tax / ($current_total - $current_discount);
                } else {
                    $discount_ratio = 0;
                    $tax_ratio = 0;
                }

                // Calculate new totals after refund
                $new_total = $current_total - $total_refund_amount;
                $new_discount = ($new_total > 0) ? $new_total * $discount_ratio : 0;
                $taxable_amount = $new_total - $new_discount;
                $new_tax = ($taxable_amount > 0) ? $taxable_amount * $tax_ratio : 0;
                $new_net_amount = $new_total - $new_discount + $new_tax;

                // Ensure values are not negative
                $new_total = max(0, $new_total);
                $new_discount = max(0, $new_discount);
                $new_tax = max(0, $new_tax);
                $new_net_amount = max(0, $new_net_amount);

                // Update original pharmacy bill basic with recalculated totals
                $bill_totals = array(
                    'total'       => round($new_total, 2),
                    'discount'    => round($new_discount, 2),
                    'tax'         => round($new_tax, 2),
                    'net_amount'  => round($new_net_amount, 2),
                );
                $this->pharmacy_model->updatePharmacyBillAmounts($bill_basic_id, $bill_totals);
            }

            $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'));
        }

        echo json_encode($array);
    }

    public function refundList($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'pharmacy');
        $data['refunds'] = $this->pharmacy_model->getAllRefunds($id);
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/refundList', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Return refund detail partial (used by AJAX modal)
     */
    public function refundDetails($id)
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }
        $data['refund'] = $this->pharmacy_model->getAllRefunds(null, $id);
        $data['details'] = $this->pharmacy_model->getRefundDetails($id);
        // render partial view
        $this->load->view('admin/pharmacy/refundDetail', $data);
    }

    public function searchMedicines()
    {
        if (!$this->rbac->hasPrivilege('medicine', 'can_view')) {
            access_denied();
        }

        $search = $this->input->get('search');
        $result = $this->pharmacy_model->search_medicine_global($search);

        echo json_encode($result);
    }

    public function payments()
    {
        if (!$this->rbac->hasPrivilege('pharmacy bill', 'can_view')) {
            access_denied();
        }

        $id = 0;
        $this->session->set_userdata('top_menu', 'pharmacy');
        $supplierCategory = $this->medicine_category_model->getSupplierCategory();
        $data["supplierCategory"] = $supplierCategory;

        // Get the filter values from POST (if form was submitted)
        $supplier_id = $this->input->post('supplier_id');
        $date_range = $this->input->post('date_range');  // Get the date_range string (e.g., '2025-12-01 to 2025-12-27')

        // Initialize date_from and date_to variables
        $date_from = '';
        $date_to = '';

        // Split the date range into date_from and date_to
        if (!empty($date_range)) {
            // Split the date_range string by ' to ' and assign to date_from and date_to
            $dates = explode(' to ', $date_range);
            if (count($dates) == 2) {
                $date_from = trim($dates[0]);
                $date_to = trim($dates[1]);
            }
        }

        // Set the supplier select value for preselecting the supplier in the dropdown
        // Only set if supplier_id was provided in POST (not empty string)
        $supplier_select = (!empty($supplier_id)) ? $supplier_id : '';

        // Get the grouped supplier bills with the filters
        // If no filters are set, this should return all records
        $data['payments'] = $this->pharmacy_model->getGroupedSupplierBills($date_from, $date_to, $supplier_id);

        // Pass the selected supplier and date range back to the view
        // These will be empty if no filters were applied, allowing the form to show empty/default values
        $data['supplier_select'] = $supplier_select;
        $data['date_range'] = (!empty($date_range)) ? $date_range : '';

        // Load views
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/paymentList', $data);
        $this->load->view('layout/footer');
    }



    public function paymentDetails($supplier_id, $date)
    {
        $data['bills'] = $this->pharmacy_model->getSupplierBillsByDate($supplier_id, $date);
        $data['date'] = $date;
        // echo "<pre>"; print_r($data);exit;
        $this->load->view('layout/header');
        $this->load->view('admin/pharmacy/paymentDetails', $data);  // Load a new view for the payment details
        $this->load->view('layout/footer');
    }

    public function savePayment()
    {
        $this->db->trans_begin();
        try {

            // Get POST data
            $supplier_id   = $this->input->post('supplier_id');
            $payment_date  = $this->input->post('payment_date');
            $paid_amounts  = $this->input->post('paid_amount');
            $payment_modes = $this->input->post('payment_mode');
            $checked_bills = $this->input->post('checked_bills');

            $total_paid_amount = 0;

            foreach ($checked_bills as $index => $bill_id) {

                if (!isset($paid_amounts[$index])) {
                    continue;
                }

                $paid_amount  = (float) $paid_amounts[$index];
                $payment_mode = $payment_modes[$index] ?? null;

                $total_paid_amount += $paid_amount;

                $billData = [
                    'paid_amount'    => $paid_amount,
                    'payment_method' => $payment_mode,
                    'payment_date'   => date('Y-m-d', strtotime($payment_date)),
                    'payment_status' => 'paid',
                ];
                $this->pharmacy_model->updateBillPayment($bill_id, $billData);


                $paymentData = [
                    'supplier_bill_basic_id' => $bill_id,
                    'payment_amount'         => $paid_amount,
                    'payment_method'         => $payment_mode,
                    'payment_date'               => date('Y-m-d'),
                ];
                $this->pharmacy_model->insertPayment($paymentData);
                if (!$this->pharmacy_model->insertPayment($paymentData)) {
                    throw new Exception('Failed to insert payment record');
                }
            }
            // echo $total_paid_amount;exit;   

            if ($total_paid_amount > 0) {

                $categoryName = 'collect stock payment';

                $exp_head_id = $this->expensehead_model->getByName($categoryName);

                if (!$exp_head_id) {
                    $exp_head_id = $this->expensehead_model->add([
                        'exp_category' => $categoryName,
                    ]);
                }

                if (!$exp_head_id) {
                    throw new Exception('Expense head creation failed');
                }

                $expenseData = [
                    'exp_head_id' => $exp_head_id,
                    'name'        => $this->input->post('supplier_name'),
                    'date'        => date(
                        'Y-m-d',
                        $this->customlib->datetostrtotime($payment_date)
                    ),
                    'amount'      => $total_paid_amount,
                ];
                // echo "<pre>"; print_r($expenseData);exit;

                if (!$this->expense_model->add($expenseData)) {
                    throw new Exception('Expense insertion failed');
                }
            }

            $this->db->trans_commit();

            $this->session->set_flashdata('status', 'success');
            $this->session->set_flashdata(
                'message',
                $this->lang->line('success_message')
            );
        } catch (Exception $e) {
            $this->db->trans_rollback();

            log_message('error', 'SavePayment Error: ' . $e->getMessage());

            $this->session->set_flashdata('status', 'error');
            $this->session->set_flashdata(
                'message',
                'Something went wrong. Please try again.'
            );
        }

        redirect('admin/pharmacy/payments');
    }

    /**
     * Supplier Return Functions
     */
    public function addReturnSupplier()
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_add')) {
            access_denied();
        }

        // $this->form_validation->set_rules('date', $this->lang->line('date'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('supplier_id', $this->lang->line('supplier'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('original_purchase_id', 'Original Purchase', 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_category_id[]', $this->lang->line('medicine') . " " . $this->lang->line('category'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('medicine_name[]', $this->lang->line('medicine') . " " . $this->lang->line('name'), 'trim|required|xss_clean');
        $this->form_validation->set_rules('expiry_date[]', $this->lang->line('expiry') . " " . $this->lang->line('date'), 'required');
        $this->form_validation->set_rules('quantity[]', $this->lang->line('quantity'), 'required|numeric');
        $this->form_validation->set_rules('purchase_price[]', $this->lang->line('purchase') . " " . $this->lang->line('price'), 'required|numeric');
        $this->form_validation->set_rules('amount[]', $this->lang->line('amount'), 'required|numeric');
        $this->form_validation->set_rules('total', $this->lang->line('total'), 'required|numeric');

        if ($this->form_validation->run() == false) {
            $msg = array(
                'date'                 => form_error('date'),
                'supplier_id'          => form_error('supplier_id'),
                'original_purchase_id' => form_error('original_purchase_id'),
                'medicine_category_id' => form_error('medicine_category_id[]'),
                'medicine_name'        => form_error('medicine_name[]'),
                'expiry_date'          => form_error('expiry_date[]'),
                'quantity'             => form_error('quantity[]'),
                'purchase_price'       => form_error('purchase_price[]'),
                'total'                => form_error('total'),
                'amount'               => form_error('amount[]'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $this->db->trans_begin();
            try {
                $supplier_id   = $this->input->post('supplier_id');
                $supplier_name = $this->input->post('supplier_name');
                $bill_date     = $this->input->post("date");
                $original_purchase_id = $this->input->post('original_purchase_id');

                // Get original purchase details
                $original_purchase = $this->pharmacy_model->getindate($original_purchase_id);

                $return_no = $this->pharmacy_model->getMaxReturnId();
                if (empty($return_no)) {
                    $return_no = 0;
                }
                $return_no = $return_no + 1;

                $data = array(
                    'return_date'          => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                    'supplier_id'          => $supplier_id,
                    'supplier_name'        => $supplier_name,
                    'original_purchase_id' => $original_purchase_id,
                    'original_invoice_no'  => isset($original_purchase['invoice_no']) ? $original_purchase['invoice_no'] : '',
                    'return_no'            => $return_no,
                    'total'                => $this->input->post('total'),
                    'discount'             => $this->input->post('discount'),
                    'tax'                  => $this->input->post('tax'),
                    'net_amount'           => $this->input->post('net_amount'),
                    'note'                 => $this->input->post('note'),
                );

                $insert_id = $this->pharmacy_model->addReturnSupplier($data);

                if (isset($_FILES["file"]) && !empty($_FILES['file']['name'])) {
                    $fileInfo = pathinfo($_FILES["file"]["name"]);
                    $img_name = $insert_id . '.' . $fileInfo['extension'];
                    move_uploaded_file($_FILES["file"]["tmp_name"], "./uploads/medicine_images/" . $img_name);
                    $data_img = array('id' => $insert_id, 'file' => 'uploads/medicine_images/' . $img_name);
                    $this->pharmacy_model->addReturnSupplier($data_img);
                }

                if ($insert_id) {
                    $medicine_category_id = $this->input->post('medicine_category_id');
                    $medicine_name        = $this->input->post('medicine_name');
                    $expiry_date          = $this->input->post('expiry_date');
                    $batch_no             = $this->input->post('batch_no');
                    $batch_amount         = $this->input->post('batch_amount');
                    $mrp                  = $this->input->post('mrp');
                    $sale_rate            = $this->input->post('sale_rate');
                    $packing_qty          = $this->input->post('packing_qty');
                    $quantity             = $this->input->post('quantity');
                    $purchase_price       = $this->input->post('purchase_price');
                    $amount               = $this->input->post('amount');
                    $medicine_batch_id    = $this->input->post('medicine_batch_id');

                    $data1 = array();
                    $j     = 0;
                    foreach ($medicine_name as $key => $mvalue) {
                        $expdate = $expiry_date[$j];
                        $explore = explode("/", $expdate);
                        $monthary = $explore[0];
                        $yearary  = $explore[1];
                        $month    = $monthary;
                        $month_number = $this->convertMonthToNumber($month);
                        $insert_date  = $yearary . "-" . $month_number . "-01";

                        $details = array(
                            'supplier_stock_return_basic_id' => $insert_id,
                            'medicine_batch_details_id' => isset($medicine_batch_id[$j]) ? $medicine_batch_id[$j] : null,
                            'pharmacy_id'              => $medicine_name[$j],
                            'medicine_category_id'     => $medicine_category_id[$j],
                            'expiry_date'              => $expiry_date[$j],
                            'expiry_date_format'       => $insert_date,
                            'batch_no'                 => isset($batch_no[$j]) && !empty($batch_no[$j]) ? $batch_no[$j] : '',
                            'batch_amount'             => isset($batch_amount[$j]) ? $batch_amount[$j] : 0,
                            'mrp'                      => $mrp[$j],
                            'sale_rate'                => $sale_rate[$j],
                            'packing_qty'              => isset($packing_qty[$j]) ? $packing_qty[$j] : 0,
                            'quantity'                 => $quantity[$j],
                            'purchase_price'           => $purchase_price[$j],
                            'amount'                   => $amount[$j],
                            'return_date'              => date('Y-m-d H:i:s', $this->customlib->datetostrtotime($bill_date)),
                        );
                        $data1[] = $details;

                        // Reduce stock from medicine_batch_details
                        if (!empty($medicine_batch_id[$j])) {
                            $this->pharmacy_model->updateMedicineBatchForReturn($medicine_batch_id[$j], $quantity[$j]);
                        }

                        $j++;
                    }

                    $this->pharmacy_model->addReturnDetail($data1);

                    // Handle expense adjustment for return - only if payments exist for this supplier bill
                    $return_amount = abs($this->input->post('net_amount'));

                    // Check if payments exist for this supplier bill in supplier_bill_payments table
                    $total_payments = $this->pharmacy_model->getSupplierBillPaymentsTotal($original_purchase_id);

                    if ($total_payments > 0) {
                        // Payments exist, so we need to deduct from expense (create negative expense entry)
                        $expense_head_name = 'Stock Return';
                        $exp_head_id = $this->expensehead_model->getByName($expense_head_name);
                        if (!$exp_head_id) {
                            $exp_head_id = $this->expensehead_model->add([
                                'exp_category' => $expense_head_name,
                            ]);
                        }

                        if ($exp_head_id) {
                            $expenseData = [
                                'exp_head_id' => $exp_head_id,
                                'name'        => $supplier_name . ' - Return',
                                'date'        => date('Y-m-d', $this->customlib->datetostrtotime($bill_date)),
                                'amount'      => -$return_amount, // Negative amount for return (deduct from expense)
                                'invoice_no'  => 'RET-' . $return_no,
                                'note'        => 'Stock return for purchase invoice: ' . (isset($original_purchase['invoice_no']) ? $original_purchase['invoice_no'] : ''),
                            ];
                            $this->expense_model->add($expenseData);
                        }
                    }
                    // If no payments exist, don't create any expense entry
                }

                $this->db->trans_commit();
                $array = array('status' => 'success', 'error' => '', 'message' => $this->lang->line('success_message'), 'insert_id' => $insert_id);
            } catch (Exception $e) {
                $this->db->trans_rollback();
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Error processing return: ' . $e->getMessage());
            }
        }
        echo json_encode($array);
    }

    public function getPurchaseBatches()
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $purchase_id = $this->input->post("purchase_id");
        $result = $this->pharmacy_model->getAvailableBatchesByPurchase($purchase_id);
        echo json_encode($result);
    }

    public function viewReturnDetail($id)
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_view')) {
            access_denied();
        }
        $print_details         = $this->printing_model->get('', 'pharmacy');
        $data["print_details"] = $print_details;
        $data['id']            = $id;
        if (isset($_POST['print'])) {
            $data["print"] = 'yes';
        } else {
            $data["print"] = 'no';
        }

        $result         = $this->pharmacy_model->getSupplierReturnDetails($id);
        $data['result'] = $result;
        $detail         = $this->pharmacy_model->getAllSupplierReturnDetails($id);
        $data['detail'] = $detail;
        $this->load->view('admin/pharmacy/printReturn', $data);
    }

    public function deleteReturnSupplier($id)
    {
        if (!$this->rbac->hasPrivilege('medicine_purchase', 'can_delete')) {
            access_denied();
        }
        if (!empty($id)) {
            $this->db->trans_begin();
            try {
                $this->pharmacy_model->deleteSupplierReturn($id);
                $this->db->trans_commit();
                $array = array('status' => 'success', 'error' => '', 'message' => 'Record Deleted Successfully.');
            } catch (Exception $e) {
                $this->db->trans_rollback();
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Error deleting return: ' . $e->getMessage());
            }
        } else {
            $array = array('status' => 'fail', 'error' => '', 'message' => '');
        }
        echo json_encode($array);
    }
}
