<?php

namespace App\Http\Controllers;

use DateTime;
use Illuminate\Http\Request;
use Saleh7\Zatca\CertificateBuilder;
use Saleh7\Zatca\Exceptions\CertificateBuilderException;
use Saleh7\Zatca\ZatcaAPI;
use GuzzleHttp\Psr7\Response;
use Saleh7\Zatca\Api\ComplianceCertificateResult;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Saleh7\Zatca\Exceptions\ZatcaException;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Saleh7\Zatca\Exceptions\ZatcaApiException;
use Saleh7\Zatca\Exceptions\ZatcaStorageException;
use Saleh7\Zatca\Mappers\InvoiceMapper;
use Saleh7\Zatca\Helpers\Certificate;
use Saleh7\Zatca\InvoiceSigner;
use Saleh7\Zatca\{
    SignatureInformation,
    UBLDocumentSignatures,
    ExtensionContent,
    UBLExtension,
    UBLExtensions,
    Signature,
    InvoiceType,
    AdditionalDocumentReference,
    TaxScheme,
    PartyTaxScheme,
    Address,
    LegalEntity,
    Delivery,
    Party,
    PaymentMeans,
    TaxCategory,
    AllowanceCharge,
    TaxSubTotal,
    TaxTotal,
    LegalMonetaryTotal,
    ClassifiedTaxCategory,
    Item,
    Price,
    InvoiceLine,
    GeneratorInvoice,
    Invoice,
    UnitCode,
    OrderReference,
    BillingReference,
    Contract,
    Attachment
};

class HomeController extends Controller
{

    public function generateCSR()
    {
        try {
            (new CertificateBuilder())
                ->setOrganizationIdentifier('310724367700003') // The Organization Identifier must be 15 digits, starting andending with 3
                // string $solutionName .. The solution provider name
                // string $model .. The model of the unit the stamp is being generated for
                // string $serialNumber .. # If you have multiple devices each should have a unique serial number
                ->setSerialNumber('SME', 'SME', '3-83f5993c-2113-434b-a66a-9b2fcc1313f3')
                ->setCommonName('SME') // The common name to be used in the certificate
                ->setCountryName('SA') // The Country name must be Two chars only
                ->setOrganizationName('SME') // The name of your organization
                ->setOrganizationalUnitName('SME') // A subunit in your organizatio
                ->setAddress('Riyadh 1234 Street') // like Riyadh 1234 Street 
                ->setInvoiceType(1100) // # Four digits, each digit acting as a bool. The order is as follows: Standard Invoice, Simplified, future use, future use 
                ->setProduction(true) // true = Production |  false = Testing
                ->setBusinessCategory('Technology') // Your business category like food, real estate, etc

                ->generateAndSave('output/certificate.csr', 'output/private.pem');

            echo "Certificate and private key saved.\n";
        } catch (CertificateBuilderException $e) {
            echo "Error: " . $e->getMessage() . "\n";
            exit(1);
        }
    }


    public function compliance()
    {
        $otp        = '305132';
        $csrPath    = public_path('output/certificate.csr');
        $jsonRelative = 'output/ZATCA_certificate_data.json';
        $jsonPath     = public_path(path: $jsonRelative);

        if (! File::exists($csrPath)) {
            return response()->json(['error' => 'CSR file not found', 'path' => $csrPath], 404);
        }

        if (File::exists($jsonPath)) {
            $payload = json_decode(File::get($jsonPath), true);
            return response()->json([
                'message'    => 'Using existing compliance result (idempotent).',
                'public_url' => url(path: $jsonRelative),
                'data'       => $payload,
            ]);
        }

        try {
            $api = new ZatcaAPI('simulation');
            $csr = File::get($csrPath);

            $res = $api->requestComplianceCertificate($csr, $otp);

            // ensure dir
            File::ensureDirectoryExists(public_path('output'), 0755, true);

            $payload = [
                'certificate' => $res->getCertificate(),
                'secret'      => $res->getSecret(),
                'requestId'   => $res->getRequestId(),
            ];

            File::put($jsonPath, json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR));

            return response()->json([
                'message'    => 'Compliance generated.',
                'public_url' => url($jsonRelative),
                'data'       => $payload,
            ], 201);
        } catch (ClientException $e) {
            $status = $e->getResponse()?->getStatusCode();
            $body   = (string) ($e->getResponse()?->getBody() ?? '');
            Log::warning('ZATCA client error', ['status' => $status, 'body' => $body]);

            if ($status === 409) {
                if (File::exists($jsonPath)) {
                    $payload = json_decode(File::get($jsonPath), true);
                    return response()->json([
                        'message'    => 'Compliance was already generated earlier (409). Returning saved result.',
                        'public_url' => url($jsonRelative),
                        'data'       => $payload,
                    ], 200);
                }

                return response()->json([
                    'error'   => 'Compliance transaction already submitted.',
                    'hint'    => 'Reuse previously saved certificate/secret, or generate a NEW CSR and use a fresh OTP.',
                    'details' => $body,
                ], 409);
            }

            return response()->json([
                'error'   => 'ZATCA API error',
                'status'  => $status,
                'details' => $body,
            ], 502);
        } catch (\Throwable $e) {
            Log::error('Compliance flow failed', ['msg' => $e->getMessage()]);
            return response()->json([
                'error'   => 'Unexpected error',
                'message' => config('app.debug') ? $e->getMessage() : 'Something went wrong.',
            ], 500);
        }
    }

    public function generateInvoice()
    {
        $csrPath    = public_path('output/certificate.csr');
        $csr = File::get($csrPath);

        // --- Signature Information & UBL Document Signatures ---
        $signatureInfo = (new SignatureInformation)
            ->setReferencedSignatureID("urn:oasis:names:specification:ubl:signature:Invoice")
            ->setID('urn:oasis:names:specification:ubl:signature:1');

        $ublDocSignatures = (new UBLDocumentSignatures)
            ->setSignatureInformation($signatureInfo);

        $extensionContent = (new ExtensionContent)
            ->setUBLDocumentSignatures($ublDocSignatures);

        $ublExtension = (new UBLExtension)
            ->setExtensionURI('urn:oasis:names:specification:ubl:dsig:enveloped:xades')
            ->setExtensionContent($extensionContent);

        // Default UBL Extensions Default
        $ublExtensions = (new UBLExtensions)
            ->setUBLExtensions([$ublExtension]);

        // --- Signature Default ---
        $signature = (new Signature)
            ->setId("urn:oasis:names:specification:ubl:signature:Invoice")
            ->setSignatureMethod("urn:oasis:names:specification:ubl:dsig:enveloped:xades");

        // --- Invoice Type ---
        $invoiceType = (new InvoiceType())
            ->setInvoice('simplified') // 'standard' or 'simplified'
            ->setInvoiceType('invoice') // 'invoice', 'debit', or 'credit', 'prepayment'
            ->setIsThirdParty(false)
            ->setIsNominal(false)
            ->setIsExportInvoice(false)
            ->setIsSummary(false)
            ->setIsSelfBilled(false);

        // add Attachment
        $attachment = (new Attachment())
            ->setBase64Content(
                'z6y6+N3kwdsXbMnbes3UJAddwo/qNyK8XkTHV96Tsmg=',
                'base64',
                'text/plain'
            );

        // --- Additional Document References ---
        $additionalDocs = [];

        // icv = Invoice counter value
        $additionalDocs[] = (new AdditionalDocumentReference())
            ->setId('ICV')
            ->setUUID("10"); //Invoice counter value

        // pih = Previous Invoice Hash
        $additionalDocs[] = (new AdditionalDocumentReference())
            ->setId('PIH')
            ->setAttachment($attachment); // Previous Invoice Hash

        // qr = QR Code Default
        $additionalDocs[] = (new AdditionalDocumentReference())
            ->setId('QR');

        // --- Tax Scheme & Party Tax Schemes ---
        $taxScheme = (new TaxScheme())
            ->setId("VAT");

        // --- Legal Entity Company ---
        $legalEntityCompany = (new LegalEntity())
            ->setRegistrationName('Maximum Speed Tech Supply');

        // --- Party Tax Scheme Company ---
        $partyTaxSchemeCompany = (new PartyTaxScheme())
            ->setTaxScheme($taxScheme)
            ->setCompanyId('399999999900003');

        // --- Address Company ---
        $addressCompany = (new Address())
            ->setStreetName('Prince Sultan')
            ->setBuildingNumber("2322")
            ->setCitySubdivisionName('Al-Murabba')
            ->setCityName('Riyadh')
            ->setPostalZone('23333')
            ->setCountry('SA');

        // --- Supplier Company ---
        $supplierCompany = (new Party())
            ->setPartyIdentification("1010010000")
            ->setPartyIdentificationId("CRN")
            ->setLegalEntity($legalEntityCompany)
            ->setPartyTaxScheme($partyTaxSchemeCompany)
            ->setPostalAddress($addressCompany);


        // --- Legal Entity Customer ---
        $legalEntityCustomer = (new LegalEntity())
            ->setRegistrationName('Fatoora Samples');

        // --- Party Tax Scheme Customer ---
        $partyTaxSchemeCustomer = (new PartyTaxScheme())
            ->setTaxScheme($taxScheme)
            ->setCompanyId('399999999800003');

        // --- Address Customer ---
        $addressCustomer = (new Address())
            ->setStreetName('Salah Al-Din')
            ->setBuildingNumber("1111")
            ->setCitySubdivisionName('Al-Murooj')
            ->setCityName('Riyadh')
            ->setPostalZone('12222')
            ->setCountry('SA');

        // --- Supplier Customer ---
        $supplierCustomer = (new Party())
            ->setLegalEntity($legalEntityCustomer)
            ->setPartyTaxScheme($partyTaxSchemeCustomer)
            ->setPostalAddress($addressCustomer);

        // --- Payment Means ---
        $paymentMeans = (new PaymentMeans())
            ->setPaymentMeansCode("10");


        // --- array of Tax Category Discount ---
        $taxCategoryDiscount = [];

        // --- Tax Category Discount ---
        $taxCategoryDiscount[] = (new TaxCategory())
            ->setPercent(15)
            ->setTaxScheme($taxScheme);

        // --- Allowance Charge (for Invoice Line) ---
        $allowanceCharges = [];
        $allowanceCharges[] = (new AllowanceCharge())
            ->setChargeIndicator(false)
            ->setAllowanceChargeReason('discount')
            ->setAmount(0.00)
            ->setTaxCategory($taxCategoryDiscount); // Tax Category Discount

        // --- Tax Category ---
        $taxCategorySubTotal = (new TaxCategory())
            ->setPercent(15)
            ->setTaxScheme($taxScheme);

        // --- Tax Sub Total ---
        $taxSubTotal = (new TaxSubTotal())
            ->setTaxableAmount(4)
            ->setTaxAmount(0.6)
            ->setTaxCategory($taxCategorySubTotal);

        // --- Tax Total ---
        $taxTotal = (new TaxTotal())
            ->addTaxSubTotal($taxSubTotal)
            ->setTaxAmount(0.6);

        // --- Legal Monetary Total ---
        $legalMonetaryTotal = (new LegalMonetaryTotal())
            ->setLineExtensionAmount(4) // Total amount of the invoice
            ->setTaxExclusiveAmount(4) // Total amount without tax
            ->setTaxInclusiveAmount(4.60) // Total amount with tax
            ->setPrepaidAmount(0) // Prepaid amount
            ->setPayableAmount(4.60) // Amount to be paid
            ->setAllowanceTotalAmount(0); // Total amount of allowances

        // --- Classified Tax Category ---
        $classifiedTax = (new ClassifiedTaxCategory())
            ->setPercent(15)
            ->setTaxScheme($taxScheme);

        // --- Item (Product) ---
        $productItem = (new Item())
            ->setName('Product') // Product name
            ->setClassifiedTaxCategory($classifiedTax); // Classified tax category

        // --- Allowance Charge (for Price) ---
        $allowanceChargesPrice = [];
        $allowanceChargesPrice[] = (new AllowanceCharge())
            ->setChargeIndicator(true)
            ->setAllowanceChargeReason('discount')
            ->setAmount(0.00);

        // --- Price ---
        $price = (new Price())
            ->setUnitCode(UnitCode::UNIT)
            ->setAllowanceCharges($allowanceChargesPrice)
            ->setPriceAmount(2);

        // --- Invoice Line Tax Total ---
        $lineTaxTotal = (new TaxTotal())
            ->setTaxAmount(0.60)
            ->setRoundingAmount(4.60);

        // --- Invoice Line(s) ---
        $invoiceLine = (new InvoiceLine())
            ->setUnitCode("PCE")
            ->setId(1)
            ->setItem($productItem)
            ->setLineExtensionAmount(4)
            ->setPrice($price)
            ->setTaxTotal($lineTaxTotal)
            ->setInvoicedQuantity(2);
        $invoiceLines = [$invoiceLine];


        // Invoice
        $invoice = (new Invoice())
            ->setUBLExtensions($ublExtensions)
            ->setUUID('3cf5ee18-ee25-44ea-a444-2c37ba7f28be')
            ->setId('SME00023')
            ->setIssueDate(new DateTime('2025-08-11 17:41:08'))
            ->setIssueTime(new DateTime('2025-08-12 17:41:08'))
            ->setInvoiceType($invoiceType)
            ->setNote('ABC')->setlanguageID('ar')
            ->setInvoiceCurrencyCode('SAR') // Currency code (ISO 4217)
            ->setTaxCurrencyCode('SAR') // Tax currency code (ISO 4217)
            ->setAdditionalDocumentReferences($additionalDocs) // Additional document references
            ->setAccountingSupplierParty($supplierCompany) // Supplier company
            ->setAccountingCustomerParty($supplierCustomer) // Customer company
            ->setPaymentMeans($paymentMeans) // Payment means
            ->setAllowanceCharges($allowanceCharges) // Allowance charges
            ->setTaxTotal($taxTotal) // Tax total
            ->setLegalMonetaryTotal($legalMonetaryTotal) // Legal monetary total
            ->setInvoiceLines($invoiceLines) // Invoice lines
            ->setSignature($signature);


        try {
            // Generate the XML (default currency 'SAR')
            // Save the XML to an output file
            GeneratorInvoice::invoice($invoice)->saveXMLFile('GeneratorSimplified_Invoice.xml');
            echo "Invoice generated successfully!";
        } catch (\Exception $e) {
            // Log error message and exit
            echo "An error occurred: " . $e->getMessage() . "\n";
            exit(1);
        }
    }
}
