<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Saleh7\Zatca\CertificateBuilder;
use Saleh7\Zatca\Exceptions\CertificateBuilderException;

class ZatcaPhase2Controller extends Controller
{
    public function generateCSRAndCompliance(Request $request)
    {
        try {
            // Validate request
            $validated = $request->validate([
                'locationId'               => 'required|integer',
                'userId'                   => 'required|integer',
                'organizationIdentifier'   => 'required|regex:/^3\d{13}$/',
                'serialNumber'             => 'required|string',
                'commonName'               => 'required|string',
                'countryName'              => 'required|string|size:2',
                'organizationName'         => 'required|string',
                'organizationalUnitName'   => 'required|string',
                'address'                  => 'required|string',
                'invoiceType'              => 'required|integer|digits:4',
                'environment'              => 'required|in:simulation,production',
                'businessCategory'         => 'required|string',
                'otp'                      => 'required|string|min:6|max:6'
            ]);
        } catch (ValidationException $e) {
            // echo "es";exit;

            return response()->json([
                'error'   => 'Validation failed',
                'details' => $e->errors()
            ], 422);
        }
        $fileBaseName     = "certificate_{$validated['locationId']}_{$validated['userId']}";
        $certificatePath  = public_path("output/{$fileBaseName}.csr");
        $privateKeyPath   = public_path("output/private_{$validated['locationId']}_{$validated['userId']}.pem");
        $jsonFileName     = "ZATCA_certificate_data_{$validated['locationId']}_{$validated['userId']}.json";
        $jsonPath         = public_path("output/{$jsonFileName}");

        try {
            \File::ensureDirectoryExists(public_path('output'), 0755, true);

            (new CertificateBuilder())
                ->setOrganizationIdentifier($validated['organizationIdentifier'])
                ->setSerialNumber(
                    $validated['commonName'],
                    $validated['organizationName'],
                    $validated['serialNumber']
                )
                ->setCommonName($validated['commonName'])
                ->setCountryName($validated['countryName'])
                ->setOrganizationName($validated['organizationName'])
                ->setOrganizationalUnitName($validated['organizationalUnitName'])
                ->setAddress($validated['address'])
                ->setInvoiceType($validated['invoiceType'])
                ->setProduction($validated['environment'] === 'production')
                ->setBusinessCategory($validated['businessCategory'])
                ->generateAndSave($certificatePath, $privateKeyPath);

            $api = new \Saleh7\Zatca\ZatcaAPI($validated['environment']);
            $csr = \File::get($certificatePath);
            $res = $api->requestComplianceCertificate($csr, $validated['otp']);

            $payload = [
                'certificate' => $res->getCertificate(),
                'secret'      => $res->getSecret(),
                'requestId'   => $res->getRequestId(),
            ];

            \File::put($jsonPath, json_encode(
                $payload,
                JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR
            ));

            return response()->json([
                'message'           => 'CSR and Compliance generated successfully.',
                'certificate_path'  => $certificatePath,
                'private_key_path'  => $privateKeyPath,
                'compliance_url'    => url("output/{$jsonFileName}"),
                'compliance_data'   => $payload,
            ], 201);
        } catch (\Throwable $e) {
            \Log::error('CSR + Compliance failed', [
                'msg'   => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'error'   => 'Unexpected error',
                'message' => config('app.debug') ? $e->getMessage() : 'Something went wrong.'
            ], 500);
        }
    }
}
